package ru.yandex.direct.jobs.redirects;

import java.time.Duration;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.common.lettuce.LettuceExecuteException;
import ru.yandex.direct.utils.HashingUtils;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE_CACHE;

/**
 * Кеширование результатов проверки редиректов с помощью Redis
 */
@Service
@ParametersAreNonnullByDefault
public class RedirectCacheService {
    private static final Logger logger = LoggerFactory.getLogger(RedirectCacheService.class);

    static final Duration EXPIRATION_TIME = Duration.ofDays(2);

    private static final Pattern URL_PATTERN = Pattern.compile("^(.*?)\\?(.*)");
    private static final Pattern PARAMETERS_TO_DELETE =
            Pattern.compile("^(utm_|openstat|match_type|matched_keyword).*$");

    private final LettuceConnectionProvider lettuceCache;

    public RedirectCacheService(@Qualifier(LETTUCE_CACHE) LettuceConnectionProvider lettuceCache) {
        this.lettuceCache = lettuceCache;
    }

    @Nullable
    public RedirectCacheRecord getFromCache(String href) {
        String normalizedHref = normalizeHref(href);
        String key = generateKey(normalizedHref);
        String jsonValue = redisGet(key);
        return Optional.ofNullable(jsonValue)
                .map(v -> JsonUtils.fromJson(v, RedirectCacheRecord.class))
                .filter(v -> Objects.equals(v.getHref(), normalizedHref)) // могут быть коллизии
                .orElse(null);
    }

    public void saveToCache(RedirectCacheRecord cacheRecord) {
        String href = cacheRecord.getHref();
        String normalizedHref = normalizeHref(href);
        cacheRecord.withHref(normalizedHref);

        String key = generateKey(normalizedHref);
        String jsonValue = JsonUtils.toJson(cacheRecord);
        redisSetex(key, jsonValue);
    }

    private String generateKey(String href) {
        // чтобы ключи не были слишком длинными используем md5
        String md5Hash = HashingUtils.getMd5HashUtf8AsHexString(href);
        return String.format("redirect-cache-%s", md5Hash);
    }

    @Nullable
    private String redisGet(String key) {
        try {
            return lettuceCache.call("redis:get", cmd -> cmd.get(key));
        } catch (LettuceExecuteException e) {
            logger.warn("error in redisGet", e);
            return null;
        }
    }

    private void redisSetex(String key, String value) {
        try {
            lettuceCache.call("redis:setex", cmd -> cmd.setex(key, EXPIRATION_TIME.getSeconds(), value));
        } catch (LettuceExecuteException e) {
            logger.warn("error in redisSetex", e);
        }
    }

    /**
     * удаляем незначимые параметры из урла (чтобы кеш работал лучше): utm_*, openstat
     * запросы делаются с полными параметрами, но для кеша урлы "нормализуются"
     */
    static String normalizeHref(String href) {
        Matcher matcher = URL_PATTERN.matcher(href);
        if (matcher.matches()) {
            String urlWithoutParameters = matcher.group(1);
            String parameters = matcher.group(2);

            String filteredParameters = Arrays.stream(parameters.split("[&;]"))
                    .filter(nameAndValue -> !PARAMETERS_TO_DELETE.matcher(nameAndValue).matches())
                    .collect(joining("&"));

            return urlWithoutParameters + (filteredParameters.isEmpty() ? "" : ("?" + filteredParameters));
        }
        return href;
    }
}
