package ru.yandex.direct.jobs.redirects;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.BannerWithAggregatorDomain;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.type.href.BannerDomainRepository;
import ru.yandex.direct.core.entity.banner.type.href.BannerUrlCheckService;
import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.domain.repository.DomainRepository;
import ru.yandex.direct.core.entity.domain.service.AggregatorDomainsService;
import ru.yandex.direct.core.entity.redirectcheckqueue.model.CheckRedirectTask;
import ru.yandex.direct.core.entity.redirectcheckqueue.model.CheckRedirectTaskResult;
import ru.yandex.direct.core.entity.redirectcheckqueue.repository.RedirectCheckQueueRepository;
import ru.yandex.direct.core.service.urlchecker.RedirectCheckResult;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.ThreadUtils;

import static org.apache.commons.lang3.StringUtils.reverse;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.StringConstraints.isValidHref;

@Service
@ParametersAreNonnullByDefault
public class RedirectCheckService {
    private static final Logger logger = LoggerFactory.getLogger(RedirectCheckService.class);

    private static final int MAX_TASKS_TO_PROCESS = 100_000;
    static final int MIN_CHUNK_SIZE = 50;
    static final int MAX_CHUNK_SIZE = 2000;
    static final int MAX_CHUNKS_TO_PROCESS = 10;
    private static final int UPDATE_CHUNK_SIZE = 500;
    private static final int MAX_THREADS = 20;
    private static final Duration WORKER_TIME_LIMIT = Duration.ofSeconds(270);

    private final BannersUrlHelper bannersUrlHelper;
    private final RedirectCheckQueueRepository redirectCheckQueueRepository;
    private final BannerUrlCheckService bannerUrlCheckService;
    private final BannerDomainRepository bannerDomainRepository;
    private final DomainRepository domainRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final AggregatorDomainsService aggregatorDomainsService;
    private final RedirectCacheService redirectCacheService;
    private final ExecutorService executorService;


    @Autowired
    public RedirectCheckService(
            BannersUrlHelper bannersUrlHelper,
            RedirectCheckQueueRepository redirectCheckQueueRepository,
            BannerUrlCheckService bannerUrlCheckService,
            BannerDomainRepository bannerDomainRepository,
            DomainRepository domainRepository,
            BannerTypedRepository bannerTypedRepository,
            AggregatorDomainsService aggregatorDomainsService,
            RedirectCacheService redirectCacheService) {
        this.bannersUrlHelper = bannersUrlHelper;
        this.redirectCheckQueueRepository = redirectCheckQueueRepository;
        this.bannerUrlCheckService = bannerUrlCheckService;
        this.bannerDomainRepository = bannerDomainRepository;
        this.domainRepository = domainRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.aggregatorDomainsService = aggregatorDomainsService;
        this.redirectCacheService = redirectCacheService;
        // TODO: DIRECT-107949 executorService
        this.executorService = Executors.newFixedThreadPool(MAX_THREADS,
                new ThreadFactoryBuilder().setNameFormat("redirect-check-service-%d").build());
    }

    /**
     * Обработка очереди проверки редиректов
     * достает id баннеров из очереди (ppc.redirect_check_queue)
     * для ссылок из этих баннеров вычисляет редиректы (с помощью zora)
     * по полученным редиректам обновляет домен в баннере
     */
    public void processQueue(int shard, boolean trustRedirectFromAnyDomain) {
        LocalDateTime borderDateTime = LocalDateTime.now();
        List<CheckRedirectTask> enabledTasks = redirectCheckQueueRepository.getTasksOlderThan(
                shard, borderDateTime, MAX_TASKS_TO_PROCESS);

        List<CheckRedirectTask> tasksToProcess = filterList(enabledTasks, RedirectCheckService::isValidTask);
        List<Long> taskIdsToDelete = filterAndMapList(enabledTasks, t -> !isValidTask(t), CheckRedirectTask::getTaskId);
        logger.info("tasksToProcess: {}, taskIdsToDelete: {}", tasksToProcess.size(), taskIdsToDelete.size());

        // удаляем из очереди заявки на удалённые баннеры
        deleteTasksByIds(shard, taskIdsToDelete);

        List<List<CheckRedirectTask>> allChunks = partitionTasksForAsyncRun(tasksToProcess);
        logger.info("chunks_count: {}", allChunks.size());
        long startTimeInNanos = System.nanoTime();

        CompletableFuture[] futures = allChunks.stream()
                .map(chunk -> CompletableFuture.runAsync(
                        Trace.current().wrap(() -> processTasks(shard, chunk, startTimeInNanos,
                                trustRedirectFromAnyDomain)),
                        executorService))
                .toArray(CompletableFuture[]::new);
        CompletableFuture.allOf(futures).join(); // IGNORE-BAD-JOIN DIRECT-149116
    }

    private static boolean isValidTask(CheckRedirectTask task) {
        if ((task.getBannerId() == null) || (task.getUserId() == null)) {
            return false;
        }
        if (StringUtils.isBlank(task.getHref()) || !isValidHref(task.getHref())) {
            logger.error("invalid href in DB {} for bid = {}", task.getHref(), task.getBannerId());
            return false;
        }
        return true;
    }

    private void deleteTasksByIds(int shard, List<Long> taskIdsToDelete) {
        for (List<Long> chunk : Lists.partition(taskIdsToDelete, UPDATE_CHUNK_SIZE)) {
            redirectCheckQueueRepository.deleteTasksByIds(shard, chunk);
        }
    }

    /**
     * Разбивает список переданных задач на чанки,
     * (каждый из полученных чанков будет обрабатываться отдельным потоком)
     * все ссылки одного пользователя должны попасть в один чанк,
     * размер каждого чанка (кроме последнего) не меньше MIN_CHUNK_SIZE и не больше MAX_CHUNK_SIZE
     * возвращает не более MAX_CHUNKS_TO_PROCESS чанков
     * (остальные задачи не будут обработаны job)
     */
    static List<List<CheckRedirectTask>> partitionTasksForAsyncRun(List<CheckRedirectTask> tasks) {
        // ссылки одного пользователя простукиваем только в один поток, чтобы не положить сайт пользователя
        Map<Long, List<CheckRedirectTask>> tasksByUserId = StreamEx.of(tasks)
                .groupingBy(CheckRedirectTask::getUserId);
        List<Long> userIds = StreamEx.of(tasks).map(CheckRedirectTask::getUserId).distinct().toList();

        List<CheckRedirectTask> currentChunk = new ArrayList<>();
        List<List<CheckRedirectTask>> allChunks = new ArrayList<>();

        for (Long userId : userIds) {
            currentChunk.addAll(tasksByUserId.get(userId));

            if (currentChunk.size() >= MIN_CHUNK_SIZE) {
                allChunks.add(currentChunk);
                currentChunk = new ArrayList<>();
            }
        }
        if (!currentChunk.isEmpty()) {
            allChunks.add(currentChunk);
        }

        // ограничиваем количество задач в одном чанке и количество чанков
        return StreamEx.of(allChunks)
                .map(chunk -> StreamEx.of(chunk).limit(MAX_CHUNK_SIZE).toList())
                .limit(MAX_CHUNKS_TO_PROCESS)
                .toList();
    }

    private void processTasks(int shard, List<CheckRedirectTask> tasks, long startTimeInNanos,
                              boolean trustRedirectFromAnyDomain) {
        Map<String, List<CheckRedirectTask>> tasksByHref = StreamEx.of(tasks)
                .groupingBy(CheckRedirectTask::getHref);
        logger.info("tasks_count: {}, hrefs_count: {}", tasks.size(), tasksByHref.keySet().size());

        List<CheckRedirectTaskResult> taskResults = new ArrayList<>();

        for (String href : tasksByHref.keySet()) {
            long elapsedTime = System.nanoTime() - startTimeInNanos;
            if (elapsedTime >= WORKER_TIME_LIMIT.toNanos()) {
                break;
            }
            try {
                RedirectCheckResult redirectCheckResult = cachedGetRedirect(href, trustRedirectFromAnyDomain);
                List<CheckRedirectTask> tasksWithThisHref = tasksByHref.get(href);
                taskResults.addAll(convertToTaskResults(href, redirectCheckResult, tasksWithThisHref));
            } catch (RuntimeException e) {
                logger.error("error in processTasks for href {}", href, e);
            }
        }
        logger.info("taskResults: {}", taskResults);

        for (List<CheckRedirectTaskResult> chunk : Lists.partition(taskResults, UPDATE_CHUNK_SIZE)) {
            updateBanners(shard, chunk);
            updateQueue(shard, chunk);
        }
    }

    private RedirectCheckResult cachedGetRedirect(String href, boolean trustRedirectFromAnyDomain) {
        logger.info("href: {}", href);

        RedirectCacheRecord recordFromCache = redirectCacheService.getFromCache(href);
        if (recordFromCache != null) {
            logger.info("from_dict: {}", recordFromCache.getRedirectDomain());
            return RedirectCheckResult.createSuccessResult(
                    recordFromCache.getRedirectUrl(),
                    recordFromCache.getRedirectDomain());
        }

        RedirectCheckResult redirectCheckResult = getRedirect(href, trustRedirectFromAnyDomain);
        logger.info("success: {}, redirect_domain: {}",
                redirectCheckResult.isSuccessful(), redirectCheckResult.getRedirectDomain());

        if (redirectCheckResult.isSuccessful()) {
            RedirectCacheRecord recordToCache = new RedirectCacheRecord()
                    .withHref(href)
                    .withRedirectUrl(redirectCheckResult.getRedirectUrl())
                    .withRedirectDomain(redirectCheckResult.getRedirectDomain());

            redirectCacheService.saveToCache(recordToCache);
        }
        return redirectCheckResult;
    }

    private RedirectCheckResult getRedirect(String href, boolean trustRedirectFromAnyDomain) {
        RedirectCheckResult result = bannerUrlCheckService.getRedirect(href, trustRedirectFromAnyDomain);

        // не делаем больше 10 запросов в секунду
        ThreadUtils.sleep(Duration.ofMillis(100));

        return result;
    }

    private List<CheckRedirectTaskResult> convertToTaskResults(
            String href, RedirectCheckResult redirectCheckResult, List<CheckRedirectTask> tasksWithThisHref) {

        String redirectUrl = redirectCheckResult.isSuccessful()
                ? redirectCheckResult.getRedirectUrl()
                : null;

        String redirectDomain = redirectCheckResult.isSuccessful()
                ? redirectCheckResult.getRedirectDomain()
                : bannersUrlHelper.extractHostFromHrefWithWwwOrNull(bannersUrlHelper.toUnicodeUrl(href));
        String lowerCaseRedirectDomain = ifNotNull(redirectDomain, String::toLowerCase);

        return StreamEx.of(tasksWithThisHref)
                .map(r -> new CheckRedirectTaskResult()
                        .withTaskId(r.getTaskId())
                        .withBannerId(r.getBannerId())
                        .withHref(href)
                        .withSuccess(redirectCheckResult.isSuccessful())
                        .withRedirectUrl(redirectUrl)
                        .withRedirectDomain(lowerCaseRedirectDomain))
                .toList();
    }

    private void updateBanners(int shard, List<CheckRedirectTaskResult> taskResults) {
        Map<Long, Pair<String, Domain>> bannerIdToHrefWithDomain = StreamEx.of(taskResults)
                .filter(RedirectCheckService::isRedirectDomainNotBlank)
                .mapToEntry(CheckRedirectTaskResult::getBannerId, r -> Pair.of(r.getHref(), new Domain()
                        .withDomain(r.getRedirectDomain())
                        .withReverseDomain(reverse(r.getRedirectDomain()))))
                .toMap();
        logger.info("banner_ids_for_update: {}", mapList(taskResults, CheckRedirectTaskResult::getBannerId));

        var domains = mapList(bannerIdToHrefWithDomain.values(), Pair::getRight);
        var domainToId = domainRepository.addDomains(shard, domains);
        domains.forEach(domain -> domain.setId(domainToId.get(domain.getDomain())));

        bannerDomainRepository.changeBannersDomainsAfterCheckRedirect(shard, bannerIdToHrefWithDomain);

        updateAggregatorDomains(shard, taskResults);
    }

    /**
     * на основе ссылки после редиректов (redirectUrl) вычисляем aggregator_domain
     * и обновляем значение в БД
     */
    private void updateAggregatorDomains(int shard, List<CheckRedirectTaskResult> taskResults) {
        Map<Long, CheckRedirectTaskResult> taskResultByBannerId = StreamEx.of(taskResults)
                .filter(RedirectCheckService::isRedirectDomainNotBlank)
                .mapToEntry(CheckRedirectTaskResult::getBannerId, Function.identity())
                .toMap();

        Set<Long> bannerIds = taskResultByBannerId.keySet();
        List<BannerWithAggregatorDomain> banners =
                bannerTypedRepository.getSafely(shard, bannerIds, BannerWithAggregatorDomain.class);
        List<BannerWithAggregatorDomain> bannersToUpdate = new ArrayList<>();

        for (BannerWithAggregatorDomain banner : banners) {
            CheckRedirectTaskResult taskResult = taskResultByBannerId.get(banner.getId());

            // ссылка в баннере могла измениться пока проверяли редиректы
            if (Objects.equals(banner.getHref(), taskResult.getHref())) {

                banner.setHref(taskResult.getRedirectUrl());
                banner.setDomain(taskResult.getRedirectDomain());
                bannersToUpdate.add(banner);
            }
        }
        aggregatorDomainsService.updateAggregatorDomains(shard, bannersToUpdate);
    }

    private static boolean isRedirectDomainNotBlank(CheckRedirectTaskResult taskResult) {
        if (StringUtils.isBlank(taskResult.getRedirectDomain())) {
            logger.error("blank redirectDomain {} for bid = {}",
                    taskResult.getRedirectDomain(), taskResult.getBannerId());
            return false;
        }
        return true;
    }

    private void updateQueue(int shard, List<CheckRedirectTaskResult> taskResults) {
        List<Long> failTaskIds = filterAndMapList(taskResults, t -> !t.getSuccess(),
                CheckRedirectTaskResult::getTaskId);
        List<Long> successTaskIds = filterAndMapList(taskResults, CheckRedirectTaskResult::getSuccess,
                CheckRedirectTaskResult::getTaskId);
        logger.info("failTaskIds: {}, successTaskIds: {}", failTaskIds, successTaskIds);

        redirectCheckQueueRepository.markTasksFailed(shard, failTaskIds);
        redirectCheckQueueRepository.deleteTasksByIds(shard, successTaskIds);
    }

    @PreDestroy
    public void shutdown() {
        executorService.shutdown();
    }
}
