package ru.yandex.direct.jobs.resendblockeddomains;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.resendblockeddomains.model.BadDomainsTitle;
import ru.yandex.direct.jobs.resendblockeddomains.model.BadDomainsTitleStatus;
import ru.yandex.direct.jobs.resendblockeddomains.repository.BadDomainsTitlesRepository;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerDomainFilter;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Скрипт обрабатывает записи из таблицы ppcdict:bad_domains_titles.
 * Для доменов в статусе for_disabling - находит все баннеры,
 * добавляет в bs_resync_queue, помечает запись как processed
 * Для доменов в статусе for_enabling - добавляет баннеры в resync, запись удаляет.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 40),
        tags = {DIRECT_PRIORITY_2, CheckTag.GROUP_INTERNAL_SYSTEMS})
@Hourglass(periodInSeconds = 300, needSchedule = TypicalEnvironment.class)
public class ResendBlockedDomainsJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(ResendBlockedDomainsJob.class);
    /**
     * максимальное количество доменов, обрабатываемых за один запуск
     */
    private static final int SELECT_LIMIT = 5;
    /**
     * максимальное количество баннеров, отправляемых в bs_resync за одну итерацию
     */
    private static final int RESYNC_CHUNK_SIZE = 50_000;

    private final BadDomainsTitlesRepository badDomainsTitlesRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final BsResyncService bsResyncService;
    private final ShardHelper shardHelper;

    @Autowired
    public ResendBlockedDomainsJob(BadDomainsTitlesRepository badDomainsTitlesRepository,
                                   BannerTypedRepository bannerTypedRepository, BsResyncService bsResyncService,
                                   ShardHelper shardHelper) {
        this.badDomainsTitlesRepository = badDomainsTitlesRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.bsResyncService = bsResyncService;
        this.shardHelper = shardHelper;
    }

    @Override
    public void execute() {
        badDomainsTitlesRepository.getNotProcessedBadDomainsTitles(SELECT_LIMIT).forEach(this::processDomain);
    }

    private void processDomain(BadDomainsTitle badDomainsTitle) {
        String domain = badDomainsTitle.getDomain();
        List<String> domains = List.of(domain, "www." + domain);

        List<BsResyncItem> resyncData = shardHelper.dbShards().stream()
                .map(shard -> bannerTypedRepository.getSafely(shard, bannerDomainFilter(domains), BannerWithHref.class))
                .flatMap(Collection::stream)
                .map(banner -> new BsResyncItem(BsResyncPriority.RESEND_DOMAINS_WITH_BLOCKED_TITLE,
                        banner.getCampaignId(), banner.getId(), null))
                .collect(Collectors.toList());

        logger.info("Fetched {} banners for domain {}", resyncData.size(), domain);
        for (List<BsResyncItem> chunk : Iterables.partition(resyncData, RESYNC_CHUNK_SIZE)) {
            bsResyncService.addObjectsToResync(chunk);
        }

        if (badDomainsTitle.getStatus().equals(BadDomainsTitleStatus.FOR_DISABLING)) {
            badDomainsTitlesRepository.markDisablingDomainAsProcessed(badDomainsTitle.getId());
            logger.info("Domain {} marked as processed", domain);
        } else if (badDomainsTitle.getStatus().equals(BadDomainsTitleStatus.FOR_ENABLING)) {
            badDomainsTitlesRepository.deleteInStatusForEnabling(badDomainsTitle.getId());
            logger.info("Domain {} removed from table", domain);
        }
    }
}
