package ru.yandex.direct.jobs.resendblockeddomains.repository;

import java.util.List;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppcdict.enums.BadDomainsTitlesStatus;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jobs.resendblockeddomains.model.BadDomainsTitle;
import ru.yandex.direct.jobs.resendblockeddomains.model.BadDomainsTitleStatus;

import static ru.yandex.direct.dbschema.ppcdict.tables.BadDomainsTitles.BAD_DOMAINS_TITLES;

@Repository
public class BadDomainsTitlesRepository {

    private final DslContextProvider dslContextProvider;

    @Autowired
    public BadDomainsTitlesRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Возвращает список доменов, чей статус не отключен - не processed
     */
    public List<BadDomainsTitle> getNotProcessedBadDomainsTitles(int selectLimit) {
        return getNotProcessedBadDomainsTitles(dslContextProvider.ppcdict(), selectLimit);
    }

    List<BadDomainsTitle> getNotProcessedBadDomainsTitles(DSLContext dslContext, int selectLimit) {
        return dslContext
                .select(BAD_DOMAINS_TITLES.ID, BAD_DOMAINS_TITLES.DOMAIN, BAD_DOMAINS_TITLES.STATUS)
                .from(BAD_DOMAINS_TITLES)
                .where(BAD_DOMAINS_TITLES.STATUS.ne(BadDomainsTitlesStatus.processed))
                .orderBy(BAD_DOMAINS_TITLES.ID.asc())
                .limit(selectLimit)
                .fetch(record -> new BadDomainsTitle(record.get(BAD_DOMAINS_TITLES.ID),
                        record.get(BAD_DOMAINS_TITLES.DOMAIN),
                        BadDomainsTitleStatus.fromSource(record.get(BAD_DOMAINS_TITLES.STATUS))));
    }

    /**
     * Помечает домен как обработанный, если он был в статусе для отключения - for_disabling
     */
    public int markDisablingDomainAsProcessed(Long id) {
        return markDisablingDomainAsProcessed(dslContextProvider.ppcdict(), id);
    }

    int markDisablingDomainAsProcessed(DSLContext dslContext, Long id) {
        return dslContext
                .update(BAD_DOMAINS_TITLES)
                .set(BAD_DOMAINS_TITLES.STATUS, BadDomainsTitlesStatus.processed)
                .where(BAD_DOMAINS_TITLES.ID.eq(id)
                        .and(BAD_DOMAINS_TITLES.STATUS.eq(BadDomainsTitlesStatus.for_disabling)))
                .execute();
    }

    /**
     * Удаляет домен, который был в статусе для включения - for_enabling
     */
    public int deleteInStatusForEnabling(Long id) {
        return deleteInStatusForEnabling(dslContextProvider.ppcdict(), id);
    }

    int deleteInStatusForEnabling(DSLContext dslContext, Long id) {
        return dslContext
                .deleteFrom(BAD_DOMAINS_TITLES)
                .where(BAD_DOMAINS_TITLES.ID.eq(id)
                        .and(BAD_DOMAINS_TITLES.STATUS.eq(BadDomainsTitlesStatus.for_enabling)))
                .execute();
    }

}
