package ru.yandex.direct.jobs.segment.common;

import java.time.LocalDate;
import java.time.temporal.ChronoUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.jobs.base.logdatatransfer.JugglerStatusCalculator;
import ru.yandex.direct.jobs.segment.common.result.SegmentUpdateResult;
import ru.yandex.direct.jobs.segment.log.SegmentSourceData;
import ru.yandex.direct.juggler.JugglerStatus;

public class SegmentJugglerStatusCalculator implements JugglerStatusCalculator<SegmentSourceData, SegmentUpdateResult> {

    private static final Logger logger = LoggerFactory.getLogger(SegmentJugglerStatusCalculator.class);

    private final int criticalLogGap;

    private int successfullyUpdatedSegments = 0;
    private int segmentsFailedToUpdate = 0;
    private LocalDate mostFreshLogDate = null;

    public SegmentJugglerStatusCalculator(int criticalLogGap) {
        this.criticalLogGap = criticalLogGap;
    }

    @Override
    public void addResult(SegmentSourceData sourceData, SegmentUpdateResult result) {
        int iterationSuccessful = result.getSegmentKeyToUploadResult().size();
        int iterationFailed = result.getSegmentKeysOfFailedUploads().size();
        successfullyUpdatedSegments += iterationSuccessful;
        segmentsFailedToUpdate += iterationFailed;

        LocalDate iterationMaxFreshLogDate = sourceData.getMostFreshLogDate();
        if (mostFreshLogDate == null ||
                (iterationMaxFreshLogDate != null && iterationMaxFreshLogDate.isAfter(mostFreshLogDate))) {
            mostFreshLogDate = iterationMaxFreshLogDate;
        }

        logger.info("added result of updating segments in Ya.Audience; success: {}, failed: {}, mostFreshLogDate: {};" +
                        "totalSuccess: {}, totalFailed: {}, max mostFreshLogDate: {}",
                iterationSuccessful, iterationFailed, iterationMaxFreshLogDate,
                successfullyUpdatedSegments, segmentsFailedToUpdate, mostFreshLogDate);
    }

    @Override
    public JugglerStatus getJugglerStatus() {
        if (isFailedTooManySegments() || isLogGapCritical()) {
            return JugglerStatus.CRIT;
        }
        return JugglerStatus.OK;
    }

    @Override
    public String getDescription() {
        StringBuilder errMsgBuilder = new StringBuilder();
        if (isFailedTooManySegments()) {
            errMsgBuilder.append(String.format("failed to update most segments (%s < %s)",
                    successfullyUpdatedSegments, segmentsFailedToUpdate));
        }
        if (isLogGapCritical()) {
            errMsgBuilder.append(String.format("log gap is critical, the most fresh log date is %s", mostFreshLogDate));
        }
        return errMsgBuilder.toString();
    }

    /**
     * Критично ли количество зафейлвшихся сегментов.
     */
    private boolean isFailedTooManySegments() {
        return segmentsFailedToUpdate > successfullyUpdatedSegments;
    }

    /**
     * Критично ли отставание лога
     */
    private boolean isLogGapCritical() {
        if (mostFreshLogDate == null) {
            return false;
        }

        LocalDate currentDate = LocalDate.now();
        long gap = ChronoUnit.DAYS.between(mostFreshLogDate, currentDate);
        return gap > criticalLogGap;
    }
}
