package ru.yandex.direct.jobs.segment.common.meta;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.model.InternalStatus;
import ru.yandex.direct.core.entity.adgroup.model.UsersSegment;
import ru.yandex.direct.core.entity.userssegments.repository.UsersSegmentRepository;
import ru.yandex.direct.jobs.base.logdatatransfer.MetaUpdatingStrategy;
import ru.yandex.direct.jobs.segment.common.result.SegmentUpdateResult;
import ru.yandex.direct.jobs.segment.log.SegmentSourceData;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.adgroup.model.ExternalAudienceStatus.IS_PROCESSED;
import static ru.yandex.direct.jobs.segment.common.SegmentUtils.segmentKeyExtractor;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

public class SegmentMetaUpdatingStrategy
        implements MetaUpdatingStrategy<UsersSegment, SegmentSourceData, SegmentUpdateResult> {

    private static final Logger logger = LoggerFactory.getLogger(SegmentMetaUpdatingStrategy.class);

    private final UsersSegmentRepository usersSegmentRepository;

    public SegmentMetaUpdatingStrategy(UsersSegmentRepository usersSegmentRepository) {
        this.usersSegmentRepository = usersSegmentRepository;
    }

    @Override
    public void updateMeta(int shard,
                           List<UsersSegment> segmentMeta,
                           SegmentSourceData segmentSourceData,
                           SegmentUpdateResult result) {
        if (segmentMeta.isEmpty() || !segmentSourceData.wereLogsRead()) {
            logger.info("no new data was read, return");
            return;
        }

        logger.info("last read log date: {}", segmentSourceData.getLastReadLogDate());
        logger.info("updating meta information of earlier fetched meta in ppc.video_segment_goals...");

        Map<SegmentKey, UsersSegment> segmentMetaMap = listToMap(segmentMeta, segmentKeyExtractor());
        //noinspection ConstantConditions
        LocalDateTime lastReadLogTime = segmentSourceData.getLastReadLogDate().atStartOfDay();

        updateSegmentMetaWithoutNewSourceData(shard, segmentMetaMap, segmentSourceData, lastReadLogTime);
        updateSegmentMetaOfSuccessfullyUpdatedSegments(shard, segmentMetaMap, result, lastReadLogTime);
        updateSegmentMetaOfFailedToUpdateSegments(shard, segmentMetaMap, result);

        logger.info("meta information of all earlier fetched meta successfully updated in ppc.video_segment_goals");
    }

    private void updateSegmentMetaWithoutNewSourceData(int shard,
                                                       Map<SegmentKey, UsersSegment> segmentMetaMap,
                                                       SegmentSourceData sourceData,
                                                       LocalDateTime lastReadLogTime) {
        List<UsersSegment> metaWithoutNewSourceData = EntryStream.of(segmentMetaMap)
                .filterKeys(sourceData.getFetchedSegmentsKeys()::contains)
                .filterKeys(segmentKey -> isEmpty(sourceData.getSegmentKeyToUids().get(segmentKey)))
                // если за какой-то день лог пропадет, то мы не станем менять дату на предыдущую
                .filterValues(segment -> lastReadLogTime.isAfter(segment.getLastSuccessUpdateTime()))
                .values()
                .toList();
        logger.info("setting lastSuccessUpdateTime = {} (last read log date) for " +
                        "segments without new data in logs (count = {})...",
                lastReadLogTime, metaWithoutNewSourceData.size());

        metaWithoutNewSourceData.forEach(
                segment -> {
                    segment.setLastSuccessUpdateTime(lastReadLogTime);
                    usersSegmentRepository.updateSegment(shard, segment);
                });
    }

    private void updateSegmentMetaOfSuccessfullyUpdatedSegments(int shard,
                                                                Map<SegmentKey, UsersSegment> segmentMetaMap,
                                                                SegmentUpdateResult segmentUpdateResult,
                                                                LocalDateTime lastReadLogTime) {
        logger.info("setting lastSuccessUpdateTime = {} (last read log date), externalAudienceStatus " +
                        "and some another data for segments with new data in logs (count = {})...",
                lastReadLogTime, segmentUpdateResult.getSegmentKeyToUploadResult().size());
        EntryStream.of(segmentUpdateResult.getSegmentKeyToUploadResult()).forKeyValue(
                (segmentKey, segmentUploadResult) -> {
                    UsersSegment segment = segmentMetaMap.get(segmentKey)
                            .withSegmentOwnerUid(segmentUploadResult.getSegmentOwnerUid())
                            .withExternalAudienceId(segmentUploadResult.getExternalAudienceId())
                            .withExternalAudienceStatus(segmentUploadResult.getExternalAudienceStatus())
                            .withErrorCount(0L)
                            .withInternalStatus(InternalStatus.COMPLETE)
                            .withLastSuccessUpdateTime(lastReadLogTime);
                    usersSegmentRepository.updateSegment(shard, segment);
                });
    }

    private void updateSegmentMetaOfFailedToUpdateSegments(int shard,
                                                           Map<SegmentKey, UsersSegment> segmentMetaMap,
                                                           SegmentUpdateResult segmentUpdateResult) {
        logger.info("setting errorCount and externalAudienceStatus = 'is_processed' " +
                        "for segments which update was unsuccessful (count = {})...",
                segmentUpdateResult.getSegmentKeysOfFailedUploads().size());
        StreamEx.of(segmentUpdateResult.getSegmentKeysOfFailedUploads()).forEach(
                segmentKey -> {
                    UsersSegment segment = segmentMetaMap.get(segmentKey);
                    segment
                            .withExternalAudienceStatus(IS_PROCESSED)
                            .withErrorCount(segment.getErrorCount() + 1);
                    usersSegmentRepository.updateSegment(shard, segment);
                });
    }
}
