package ru.yandex.direct.jobs.segment.common.metrica;

import java.math.BigInteger;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.stereotype.Service;

import ru.yandex.direct.audience.client.YaAudienceClient;
import ru.yandex.direct.audience.client.exception.SegmentNotModifiedException;
import ru.yandex.direct.audience.client.model.SegmentContentType;
import ru.yandex.direct.core.entity.adgroup.model.ExternalAudienceStatus;
import ru.yandex.direct.core.entity.metrika.model.Segment;
import ru.yandex.direct.core.entity.metrika.utils.YaAudienceConverter;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static ru.yandex.direct.core.entity.metrika.utils.YaAudienceConverter.convertSegment;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class UploadSegmentsService {
    public static final int MIN_SEGMENT_SIZE = 1000;

    private final YaAudienceClient yaAudienceClient;

    public UploadSegmentsService(YaAudienceClient yaAudienceClient) {
        this.yaAudienceClient = yaAudienceClient;
    }

    public List<Segment> getSegments(String ulogin) {
        var audienceSegments = yaAudienceClient.getSegments(ulogin);
        return mapList(audienceSegments, YaAudienceConverter::convertSegment);
    }

    public Segment uploadUids(Collection<BigInteger> ids, String ulogin) {
        String data = getStringFromIds(ids);
        return uploadUids(data.getBytes(), ulogin);
    }

    private Segment uploadUids(byte[] content, String ulogin) {
        try (TraceProfile profile = Trace.current().profile("uploadSegmentsService:uploadUids")) {
            var audienceSegment = yaAudienceClient.uploadSegment(ulogin, content);
            return convertSegment(audienceSegment);
        }
    }

    public Segment confirm(Long segmentId, String ulogin, String segmentName, SegmentContentType contentType) {
        var audienceSegment = yaAudienceClient.confirmSegment(ulogin, segmentId, segmentName, contentType);
        return convertSegment(audienceSegment);
    }

    private static String getStringFromIds(Collection<BigInteger> ids) {
        return ids.stream().map(BigInteger::toString).collect(Collectors.joining("\n"));
    }

    /**
     * Модификация сегмента с предохранением от того, что сегмент не изменился.
     * Если сегмент не изменился, возвратим информацию о сегменте со статусом
     * {@link ExternalAudienceStatus#IS_PROCESSED},
     * что бы он не выбирался на следующей итерации. Если бы сегмент был в процессе обработки, аудитории бы вернули
     * другую ошибку.
     */
    public Segment modifySegmentSafe(Long segmentId, String ulogin, Collection<BigInteger> newIds) {
        try {
            return modifySegment(segmentId, ulogin, newIds);
        } catch (SegmentNotModifiedException e) {
            return new Segment()
                    .withId(segmentId)
                    .withStatus(ExternalAudienceStatus.IS_PROCESSED);
        }
    }

    Segment modifySegment(Long segmentId, String ulogin, Collection<BigInteger> newIds) {
        try (TraceProfile profile = Trace.current().profile("uploadSegmentsService:modify")) {
            var audienceSegment = yaAudienceClient.modifySegment(ulogin, segmentId,
                    getStringFromIds(newIds).getBytes());
            return convertSegment(audienceSegment);
        }
    }
}
