package ru.yandex.direct.jobs.segment.common.preprocessor;

import java.math.BigInteger;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.model.UsersSegment;
import ru.yandex.direct.jobs.base.logdatatransfer.SourceDataPreprocessor;
import ru.yandex.direct.jobs.segment.common.meta.SegmentKey;
import ru.yandex.direct.jobs.segment.log.SegmentSourceData;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.jobs.segment.common.SegmentUtils.segmentKeyExtractor;
import static ru.yandex.direct.jobs.segment.common.metrica.UploadSegmentsService.MIN_SEGMENT_SIZE;
import static ru.yandex.direct.jobs.segment.log.SegmentSourceData.sourceData;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

public class FakeUidsSegmentSourceDataPreprocessor implements SourceDataPreprocessor<UsersSegment, SegmentSourceData> {

    static final int MAX_GAP_TO_CREATE = 3;
    private static final Logger logger = LoggerFactory.getLogger(FakeUidsSegmentSourceDataPreprocessor.class);
    private final FakeUidsHolder fakeUidsHolder;

    public FakeUidsSegmentSourceDataPreprocessor(FakeUidsHolder fakeUidsHolder) {
        this.fakeUidsHolder = fakeUidsHolder;
    }

    @Override
    public SegmentSourceData preprocess(List<UsersSegment> segmentMeta, SegmentSourceData segmentSourceData) {
        // если логи не читались, то ничего не обрабатываем
        if (!segmentSourceData.wereLogsRead()) {
            return segmentSourceData;
        }

        Map<SegmentKey, UsersSegment> segmentMetaMap = listToMap(segmentMeta, segmentKeyExtractor());

        Map<SegmentKey, Set<BigInteger>> newSegmentKeyToUidsMap = EntryStream.of(segmentMetaMap)
                .filterKeys(segmentSourceData.getFetchedSegmentsKeys()::contains)
                .mapToValue((segmentKey, meta) -> {
                    Set<BigInteger> uids = segmentSourceData.getUidsOrEmptySet(segmentKey);
                    return preprocess(segmentKey, meta, uids);
                })
                .toMap();

        return sourceData(segmentSourceData.getFetchedSegmentsKeys(),
                newSegmentKeyToUidsMap,
                segmentSourceData.getLastReadLogDate(),
                segmentSourceData.getMostFreshLogDate(),
                segmentSourceData.getContentType());
    }

    private Set<BigInteger> preprocess(SegmentKey segmentKey, UsersSegment segmentMeta, Set<BigInteger> uids) {
        // если прочитанных данных в логе достаточно для создания сегмента в Я.Аудиториях, то ничего не меняем
        if (uids.size() >= MIN_SEGMENT_SIZE) {
            logger.info("there are enough uids ({}) for new segment {}, no preprocessing",
                    uids.size(), segmentKey);
            return uids;
        }

        // если данные в логе есть, но их недостаточно, то досыпаем фейков
        if (!uids.isEmpty()) {
            logger.info("there are not enough uids ({}) for new segment {}, adding fakes...",
                    uids.size(), segmentKey);
            Set<BigInteger> newUids = new HashSet<>(uids);
            newUids.addAll(fakeUidsHolder.getFakeUids().subList(0, MIN_SEGMENT_SIZE - uids.size()));
            return newUids;
        }

        // если данных в логе совсем нет, и при этом запись в video_segment_goals создана достаточно давно,
        // то создаем сегмент в Я.Аудиториях из одних только фейков
        LocalDate currentDate = LocalDate.now();
        LocalDate createDate = segmentMeta.getTimeCreated().toLocalDate();
        checkState(!createDate.isAfter(currentDate),
                "segment creation date (%s) cannot be after current date", createDate);
        long gap = ChronoUnit.DAYS.between(createDate, currentDate);
        if (gap > MAX_GAP_TO_CREATE) {
            logger.info("there are no uids for new segment {}, but time gap is reached, " +
                    "adding fakes to create segment with only fakes...", segmentKey);
            Set<BigInteger> newUids = new HashSet<>(uids);
            newUids.addAll(fakeUidsHolder.getFakeUids().subList(0, MIN_SEGMENT_SIZE));
            return newUids;
        }

        logger.info("there are no uids for new segment {}, but time gap is not reached, no preprocessing", segmentKey);
        return uids;
    }
}
