package ru.yandex.direct.jobs.segment.common.target.name;

import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.common.util.LocaleGuard;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdShowType;
import ru.yandex.direct.core.entity.adgroup.model.UsersSegment;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.i18n.Translatable;

import static java.util.Collections.singletonList;

@Service
public class SegmentNameHelper {

    private static final int MAX_LENGTH = 249;
    private static final String SEGMENT_NAME_FORMAT = "%s %s-%s";

    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final UserRepository userRepository;
    private final TranslationService translationService;

    private static final Map<AdShowType, Translatable> NAMES = new EnumMap<>(AdShowType.class);
    private static final Map<Pair<AdGroupType, AdShowType>, Translatable> EXTRA_NAMES = new HashMap<>();
    private static final SegmentNameTranslations TRANSLATIONS = SegmentNameTranslations.INSTANCE;

    static {
        NAMES.put(AdShowType.START, TRANSLATIONS.start());
        NAMES.put(AdShowType.FIRST_QUARTILE, TRANSLATIONS.firstQuartile());
        NAMES.put(AdShowType.MIDPOINT, TRANSLATIONS.midpoint());
        NAMES.put(AdShowType.THIRD_QUARTILE, TRANSLATIONS.thirdQuartile());
        NAMES.put(AdShowType.COMPLETE, TRANSLATIONS.complete());

        EXTRA_NAMES.put(Pair.of(AdGroupType.BASE, AdShowType.START), TRANSLATIONS.startInPartner());
    }

    @Autowired
    public SegmentNameHelper(CampaignRepository campaignRepository,
                             AdGroupRepository adGroupRepository,
                             UserRepository userRepository,
                             TranslationService translationService) {
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.userRepository = userRepository;
        this.translationService = translationService;
    }

    public String getSegmentName(int shard, UsersSegment usersSegment) {
        Long adGroupId = usersSegment.getAdGroupId();
        CampaignSimple campaign =
                campaignRepository.getCampaignsSimpleByAdGroupIds(shard, singletonList(adGroupId)).get(adGroupId);
        AdGroupType adGroupType =
                adGroupRepository.getAdGroupsWithType(shard, null, singletonList(adGroupId)).get(adGroupId).getType();
        User user = userRepository.fetchByUids(shard, singletonList(usersSegment.getSegmentOwnerUid())).get(0);
        return getSegmentName(usersSegment, campaign, adGroupType, user.getLang());
    }

    String getSegmentName(UsersSegment segment, CampaignSimple campaign, AdGroupType adGroupType,
                          Language language) {
        try (LocaleGuard ignored = LocaleGuard.fromLanguage(language)) {
            String campaignId = translate(TRANSLATIONS.campaign()) + ":" + campaign.getId();

            String outdoorIndicator = "(" + translate(TRANSLATIONS.outdoor()) + ")";
            String adGroupPart = translate(TRANSLATIONS.group()) + ":" + segment.getAdGroupId()
                    + (adGroupType == AdGroupType.CPM_OUTDOOR ? outdoorIndicator : "");

            String adShowTypePart = translate(getAdShowTypeName(adGroupType, segment.getType()));

            String segmentNameWithoutCampaignName =
                    String.format(SEGMENT_NAME_FORMAT, campaignId, adGroupPart, adShowTypePart);
            int maxCampaignNameLength = MAX_LENGTH - segmentNameWithoutCampaignName.length() - 4;
            String campaignName = campaign.getName().length() <= maxCampaignNameLength ?
                    campaign.getName() : campaign.getName().substring(0, maxCampaignNameLength);

            String campaignPart = campaignId + "-\"" + campaignName + "\".";

            return String.format(SEGMENT_NAME_FORMAT, campaignPart, adGroupPart, adShowTypePart);
        }
    }

    private Translatable getAdShowTypeName(AdGroupType adGroupType, AdShowType adShowType) {
        return EXTRA_NAMES.getOrDefault(Pair.of(adGroupType, adShowType), NAMES.get(adShowType));
    }

    private String translate(Translatable translatable) {
        return translationService.translate(translatable);
    }
}
