package ru.yandex.direct.jobs.segment.jobs.intermediate;

import java.time.LocalDate;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.jobs.segment.log.LogTableNavigator;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.yql.ResultSetFuture;

import static ru.yandex.direct.jobs.segment.log.LogTableNavigator.IGNORE_MISSED_DAYS_TRUE;

/**
 * Из подневных табличек с логами варит подневные таблички,
 * в которых присутствуют только необходимые события из исходного лога.
 *
 * исходный лог            |    результат работы данной джобы
 * ------------------------|---------------------------------------------
 * source_path/03.12.2020  |    dest_path/03.12.2020
 * source_path/04.12.2020  |    dest_path/04.12.2020
 * source_path/05.12.2020  | -> джоба видит что таблички еще нет и создаёт dest_path/05.12.2020
 */
@ParametersAreNonnullByDefault
public class SegmentsIntermediateTablesCreator {

    private static final Logger logger = LoggerFactory.getLogger(SegmentsIntermediateTablesCreator.class);

    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final LocalDate forceDestTableStartDate;
    private final Function<LocalDate, String> sourceTablePathProvider;
    private final Function<LocalDate, String> destTablePathProvider;
    private final LogTableNavigator sourceLogTableNavigator;
    private final LogTableNavigator destLogTableNavigator;
    private final String yql;

    public SegmentsIntermediateTablesCreator(
            YtProvider ytProvider,
            YtCluster ytCluster,
            @Nullable LocalDate forceDestTableStartDate,
            Function<LocalDate, String> sourceTablePathProvider,
            Function<LocalDate, String> destTablePathProvider,
            String yqlPath,
            boolean ignoreMissedDays
    ) {
        this.ytProvider = ytProvider;
        this.ytCluster = ytCluster;
        this.forceDestTableStartDate = forceDestTableStartDate;
        this.sourceTablePathProvider = sourceTablePathProvider;
        this.destTablePathProvider = destTablePathProvider;

        this.sourceLogTableNavigator = new LogTableNavigator(ytProvider, ytCluster,
                sourceTablePathProvider, () -> ignoreMissedDays);

        this.destLogTableNavigator = new LogTableNavigator(ytProvider, ytCluster,
                destTablePathProvider,
                LogTableNavigator.LastDayMode.USE_LAST_DAY,
                IGNORE_MISSED_DAYS_TRUE);
        yql = String.join("\n",
                new ClassPathResourceInputStreamSource(yqlPath).readLines()
        );
    }

    public void execute() {
        LocalDate destTableStartDate = forceDestTableStartDate;
        if (destTableStartDate == null) {
            LocalDate mostFreshDestTableDate = destLogTableNavigator.getTheMostFreshLogDate();
            destTableStartDate = mostFreshDestTableDate.plusDays(1);
        }

        LocalDate mostFreshSourceTableDate = sourceLogTableNavigator.getTheMostFreshLogDate();
        if (destTableStartDate.isAfter(mostFreshSourceTableDate)) {
            logger.info("most fresh dest table {} already presents, exit", destTableStartDate);
            return;
        }

        LocalDate currentProcessingDate = destTableStartDate;
        while (!currentProcessingDate.isAfter(mostFreshSourceTableDate)) {
            createDestTable(currentProcessingDate);
            currentProcessingDate = currentProcessingDate.plusDays(1);
        }
    }

    private void createDestTable(LocalDate localDate) {
        String destTablePath = destTablePathProvider.apply(localDate);
        String sourceTablePath = sourceTablePathProvider.apply(localDate);
        YqlQuery yqlQuery = new YqlQuery(yql, destTablePath, sourceTablePath);

        logger.info("starting YQL for creating intermediate table `{}`...", destTablePath);
        ResultSetFuture resultSetFuture = ytProvider.getOperator(ytCluster).yqlQueryBegin(yqlQuery);
        logger.info("started YQL (operationId = {}) for creating intermediate table `{}`",
                resultSetFuture.getOperationId(), destTablePath);
    }
}
