package ru.yandex.direct.jobs.segment.jobs.status;

import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.userssegments.repository.UsersSegmentRepository;
import ru.yandex.direct.dbschema.ppc.enums.VideoSegmentGoalsExternalAudienceStatus;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod.TELEGRAM;
import static ru.yandex.direct.jobs.segment.jobs.status.SegmentYtExternalStatus.UNKNOWN;
import static ru.yandex.direct.juggler.JugglerStatus.CRIT;
import static ru.yandex.direct.juggler.JugglerStatus.OK;
import static ru.yandex.direct.juggler.JugglerStatus.WARN;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 25),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_MEXICANO,
                method = TELEGRAM,
                status = {OK, WARN, CRIT})
)
@Hourglass(cronExpression = "0 0 1 * * ?", needSchedule = ProductionOnly.class)
public class SegmentStatusUpdateJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(SegmentStatusUpdateJob.class);

    private static final int FETCH_LIMIT = 50_000;
    private static final int UPDATE_LIMIT = 1_000;

    private final UsersSegmentRepository usersSegmentRepository;
    private final AudienceSegmentSimpleYtRepository audienceSegmentSimpleYtRepository;

    @Autowired
    public SegmentStatusUpdateJob(UsersSegmentRepository usersSegmentRepository,
                                  AudienceSegmentSimpleYtRepository audienceSegmentSimpleYtRepository) {
        this.usersSegmentRepository = usersSegmentRepository;
        this.audienceSegmentSimpleYtRepository = audienceSegmentSimpleYtRepository;
    }

    @Override
    public void execute() {
        int offset = 0;
        int fetched;

        do {
            List<Long> segmentIdsChunk = usersSegmentRepository
                    .getExternalAudienceIds(getShard(), LimitOffset.limited(FETCH_LIMIT, offset));
            fetched = segmentIdsChunk.size();
            logger.info("fetched {} segment ids with offset = {}", fetched, offset);

            segmentIdsChunk = filterList(segmentIdsChunk, id -> id != 0);
            logger.info("segments with non-zero ids: {}", segmentIdsChunk.size());

            List<SegmentWithStatus> segmentStatuses = audienceSegmentSimpleYtRepository.getStatuses(segmentIdsChunk);

            checkUnknownStatuses(segmentStatuses);
            segmentStatuses = filterList(segmentStatuses, s -> s.getStatus() != UNKNOWN);
            updateStatuses(segmentStatuses);

            offset += fetched;
        } while (fetched == FETCH_LIMIT);
    }

    /**
     * Если внезапно появился незнакомый статус, посмотрите, какие статусы в табличке в YT,
     * подставив id сегментов из лога:
     * <p>
     * use hahn;
     * <p>
     * select id, status
     * from `//home/audience/production/export/segments_simple`
     * where id in (...)
     * <p>
     * Так же стоит посмотреть список всех возможных статусов в табличке и сравнить с enum-ом SegmentYtExternalStatus.
     * Про новые статусы можно спросить у Юры Галицкого, чтобы понять, в какой
     * внутренний статус их сконвертировать в методе convertExternalStatus.
     */
    private void checkUnknownStatuses(List<SegmentWithStatus> segmentStatuses) {
        List<SegmentWithStatus> segmentWithUnknownStatuses =
                filterList(segmentStatuses, s -> s.getStatus() == UNKNOWN);
        if (segmentWithUnknownStatuses.isEmpty()) {
            return;
        }

        List<Long> exampleIds;
        if (segmentWithUnknownStatuses.size() > 20) {
            exampleIds = mapList(segmentWithUnknownStatuses.subList(0, 20), SegmentWithStatus::getId);
        } else {
            exampleIds = mapList(segmentWithUnknownStatuses, SegmentWithStatus::getId);
        }
        setJugglerStatus(CRIT, String.format("unknown statuses (%s) for segments (%S), example external ids: %s" +
                        segmentWithUnknownStatuses.size(),
                segmentStatuses.size(),
                StringUtils.join(exampleIds, ", ")));
    }

    private void updateStatuses(List<SegmentWithStatus> audienceSegmentStatuses) {
        Lists.partition(audienceSegmentStatuses, UPDATE_LIMIT).forEach(chunk -> {
            Map<Long, VideoSegmentGoalsExternalAudienceStatus> externalIdToStatusMap =
                    convertStatusesToCoreFormat(chunk);
            usersSegmentRepository.updateSegmentStatusesByExternalAudienceId(getShard(), externalIdToStatusMap);
        });
    }

    private Map<Long, VideoSegmentGoalsExternalAudienceStatus> convertStatusesToCoreFormat(
            List<SegmentWithStatus> segmentStatuses) {
        return StreamEx.of(segmentStatuses)
                .mapToEntry(this::extractCoreStatus)
                .mapKeys(SegmentWithStatus::getId)
                .toMap();
    }

    private VideoSegmentGoalsExternalAudienceStatus extractCoreStatus(SegmentWithStatus segmentStatus) {
        if (segmentStatus.isDeleted()) {
            return VideoSegmentGoalsExternalAudienceStatus.deleted;
        }
        SegmentYtExternalStatus segmentYtStatus = segmentStatus.getStatus();
        if (segmentYtStatus == null) {
            throw new IllegalArgumentException("status is null for segment_id = " + segmentStatus.getId());
        }
        switch (segmentYtStatus) {
            case UPDATED:
            case UPLOADED:
            case CONFIRMED:
            case PROCESS_UPDATED:
            case IS_PROCESSED:
                return VideoSegmentGoalsExternalAudienceStatus.is_processed;
            case PROCESSED_FULLY:
                return VideoSegmentGoalsExternalAudienceStatus.processed;
            case PROCESSING_FAILED:
                return VideoSegmentGoalsExternalAudienceStatus.processing_failed;
            case FEW_DATA:
                return VideoSegmentGoalsExternalAudienceStatus.few_data;
            default:
                throw new IllegalArgumentException("unknown status: " + segmentYtStatus);
        }
    }
}
