package ru.yandex.direct.jobs.segment.log

import com.google.common.base.Preconditions
import ru.yandex.direct.jobs.segment.common.SegmentUtils.SEGMENTS_CLUSTER
import ru.yandex.direct.jobs.segment.common.SegmentUtils.intermediateTablePathProvider
import ru.yandex.direct.jobs.segment.common.SegmentUtils.intermediateTableRootPath
import ru.yandex.direct.jobs.segment.common.meta.SegmentKey
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtTable
import java.math.BigInteger
import java.sql.ResultSet
import java.time.LocalDate

abstract class BaseIntermediateSegmentYtRepository(
    private val ytProvider: YtProvider,
    intermediateTablePath: String
): IntermediateSegmentYtRepository {
    protected val intermediateTableRootPath: String
    private val logTableNavigator: LogTableNavigator
    init {
        intermediateTableRootPath = intermediateTableRootPath(ytProvider, intermediateTablePath)
        logTableNavigator = LogTableNavigator(
            ytProvider, SEGMENTS_CLUSTER,
            intermediateTablePathProvider(intermediateTableRootPath),
            LogTableNavigator.LastDayMode.USE_LAST_DAY, LogTableNavigator.IGNORE_MISSED_DAYS_TRUE
        )
    }

    companion object {
        /**
         * Максимальное количество дней, за которые храним промежуточные таблички, которые генерируются джобами
         * [ru.yandex.direct.jobs.segment.jobs.intermediate.common.CommonSegmentIntermediateTableCreateJob] и
         * [ru.yandex.direct.jobs.segment.jobs.intermediate.cpmdefault.CpmDefaultSegmentIntermediateTableCreateJob].
         */
        private const val MAX_DAYS = 15
    }

    override fun intermediateTablePathProvider() = intermediateTablePathProvider(intermediateTableRootPath)

    override fun getMostFreshLogDate() = logTableNavigator.getTheMostFreshLogDate()

    override fun getOldestLogDate() = mostFreshLogDate.minusDays((MAX_DAYS - 1).toLong())

    override fun getCount(
        segmentKeys: Collection<SegmentKey>,
        from: LocalDate,
        to: LocalDate
    ): Map<SegmentKey, Long> {
        checkEveryDayLogsExist(from, to)
        val query = getCountQuery(segmentKeys, from, to)
        val result = mutableMapOf<SegmentKey, Long>()
        ytProvider.getOperator(SEGMENTS_CLUSTER).yqlQuery(query, {
                resultSet -> mapCountResult(resultSet, result)
        })
        return result
    }

    override fun getData(
        segmentKeys: Collection<SegmentKey>,
        from: LocalDate,
        to: LocalDate
    ): Map<SegmentKey, Set<BigInteger>> {
        checkEveryDayLogsExist(from, to)
        val query = getDataQuery(segmentKeys, from, to)
        val result = mutableMapOf<SegmentKey, MutableSet<BigInteger>>()
        ytProvider.getOperator(SEGMENTS_CLUSTER).yqlQuery(query, {
                resultSet -> mapDataResult(resultSet, result)
        })
        return result
    }

    abstract fun getCountQuery(segmentKeys: Collection<SegmentKey>, from: LocalDate, to: LocalDate): String

    abstract fun getDataQuery(segmentKeys: Collection<SegmentKey>, from: LocalDate, to: LocalDate): String

    abstract fun mapCountResult(resultSet: ResultSet, result: MutableMap<SegmentKey, Long>): Any?

    abstract fun mapDataResult(resultSet: ResultSet, result: MutableMap<SegmentKey, MutableSet<BigInteger>>): Any?

    private fun checkEveryDayLogsExist(from: LocalDate, to: LocalDate) {
        Preconditions.checkArgument(!from.isAfter(to), "\"from\" date %s is after \"to\" date %s", from, to)
        var curDate = from
        while (!curDate.isAfter(to)) {
            val path = intermediateTablePathProvider().apply(curDate)
            val exists = ytProvider.getOperator(SEGMENTS_CLUSTER).exists(YtTable(path))
            Preconditions.checkState(exists, "log table doesn't exist in checked period (%s - %s): %s", from, to, path)
            curDate = curDate.plusDays(1)
        }
    }

    protected fun getCount(obj: Any): Long {
        if (obj is Long) {
            return obj
        }
        if (obj is Int) {
            return obj.toLong()
        }
        throw IllegalStateException("can't parse count from " + obj.javaClass)
    }
}
