package ru.yandex.direct.jobs.segment.log;

import java.time.LocalDate;
import java.util.Iterator;
import java.util.function.Function;
import java.util.function.Supplier;

import com.google.common.base.Suppliers;

import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.jobs.segment.common.SegmentUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.jobs.segment.common.SegmentUtils.ROW_COUNT_ATTRIBUTE_NAME;
import static ru.yandex.direct.jobs.segment.log.LogTableNavigator.LastDayMode.USE_LAST_DAY;

/**
 * Умеет находить самые свежие логи, которые можно читать (самая свежая табличка минус 1 день),
 * а так же предоставляет итератор для навигации по табличкам с логами с определенной даты.
 */
public class LogTableNavigator {

    public static final Supplier<Boolean> IGNORE_MISSED_DAYS_TRUE = () -> true;
    public static final Supplier<Boolean> IGNORE_MISSED_DAYS_FALSE = () -> false;

    static final int DAYS_BACK_LIMIT = 15;

    private final YtProvider ytProvider;
    private final YtCluster ytCluster;
    private final Function<LocalDate, String> logPathProvider;
    private final int daysBackLimit;
    private final LastDayMode lastDayMode;
    private final Supplier<Boolean> ignoreMissedDays;

    /**
     * @param ytProvider           //
     * @param ytCluster            //
     * @param logPathProvider      предоставляет путь к логам в зависимости от даты
     * @param ignoreMissedDaysProp параметр позволяет игнорировать пропущенные подневные логи;
     *                             если он выключен, то в таком случае будет сгенерировано исключение,
     *                             чтобы ситуация не осталась незамеченной. Вычитывается из базы один раз.
     */
    public LogTableNavigator(YtProvider ytProvider, YtCluster ytCluster,
                             Function<LocalDate, String> logPathProvider,
                             PpcProperty<Boolean> ignoreMissedDaysProp) {
        this(ytProvider, ytCluster, logPathProvider, () -> ignoreMissedDaysProp.getOrDefault(false));
    }

    /**
     * @param ytProvider       //
     * @param ytCluster        //
     * @param logPathProvider  предоставляет путь к логам в зависимости от даты
     * @param ignoreMissedDays параметр позволяет игнорировать пропущенные подневные логи;
     *                         если он выключен, то в таком случае будет сгенерировано исключение,
     *                         чтобы ситуация не осталась незамеченной. Вычитывается из базы один раз.
     */
    public LogTableNavigator(YtProvider ytProvider, YtCluster ytCluster,
                             Function<LocalDate, String> logPathProvider,
                             Supplier<Boolean> ignoreMissedDays) {
        this(ytProvider, ytCluster, logPathProvider, LastDayMode.SKIP_LAST_DAY, ignoreMissedDays);
    }

    /**
     * @param ytProvider       //
     * @param ytCluster        //
     * @param logPathProvider  предоставляет путь к логам в зависимости от даты
     * @param lastDayMode      как обращаться с логами за последний день
     * @param ignoreMissedDays параметр позволяет игнорировать пропущенные подневные логи;
     *                         если он выключен, то в таком случае будет сгенерировано исключение,
     *                         чтобы ситуация не осталась незамеченной. Вычитывается из базы один раз.
     */
    public LogTableNavigator(YtProvider ytProvider, YtCluster ytCluster,
                             Function<LocalDate, String> logPathProvider,
                             LastDayMode lastDayMode,
                             Supplier<Boolean> ignoreMissedDays) {
        this.ytProvider = ytProvider;
        this.ytCluster = ytCluster;
        this.logPathProvider = logPathProvider;
        this.daysBackLimit = DAYS_BACK_LIMIT;
        this.lastDayMode = lastDayMode;
        this.ignoreMissedDays = Suppliers.memoize(ignoreMissedDays::get);
    }

    /**
     * Если параметр lastDayMode == USE_LAST_DAY, то возвращает последнюю дату, на которую есть дневные логи.
     * Если параметр lastDayMode == SKIP_LAST_DAY, то возвращаемая дата отстает на 1 день от даты самых
     * свежих дневных логов, чтобы гарантировать, что читаемые логи полные и в дальнейшем их не нужно перечитывать.
     */
    public LocalDate getTheMostFreshLogDate() {
        YtOperator ytOperator = SegmentUtils.getYtOperator(ytProvider, ytCluster);
        LocalDate curDate = LocalDate.now();
        boolean firstTableFound = false;
        for (int daysBack = 0; daysBack <= DAYS_BACK_LIMIT; daysBack++) {
            YtTable table = new YtTable(logPathProvider.apply(curDate));
            boolean tableExists = checkTableExistence(ytOperator, table);
            if (tableExists) {

                if (lastDayMode == USE_LAST_DAY) {
                    return curDate;
                }

                if (firstTableFound) {
                    return curDate;
                } else {
                    firstTableFound = true;
                }
            } else {
                checkState(ignoreMissedDays.get() || !firstTableFound,
                        "log table for date %s doesn't exist, but more fresh log table found", curDate);
            }
            curDate = curDate.minusDays(1);
        }
        throw new IllegalStateException(String.format("there are no logs for last %s days", daysBackLimit));
    }

    /**
     * Возвращает итератор по подневным логам с направлением от старых логов к новым.
     *
     * @param startDate дата лога, с которого необходимо получить итератор.
     * @param maxDays   максимальное количество дней в итераторе
     */
    public Iterator<DayLog> getDayLogsIterator(LocalDate startDate, int maxDays) {
        checkArgument(maxDays > 0, "maxDays must be greater than zero");

        LocalDate mostFreshLogDate = getTheMostFreshLogDate();
        LocalDate limitedFinishLogDate = startDate.plusDays(maxDays - 1);

        LocalDate finishDate = mostFreshLogDate.isBefore(limitedFinishLogDate) ?
                mostFreshLogDate : limitedFinishLogDate;

        return new Iterator<>() {
            private static final String ERROR = "log table for date %s doesn't exist";

            private YtOperator ytOperator = SegmentUtils.getYtOperator(ytProvider, ytCluster);
            private LocalDate curDate = startDate.minusDays(1);

            @Override
            public boolean hasNext() {
                while (curDate.isBefore(finishDate)) {
                    LocalDate nextDay = curDate.plusDays(1);
                    YtTable nextDayLogTable = new YtTable(logPathProvider.apply(nextDay));
                    if (checkTableExistence(ytOperator, nextDayLogTable)) {
                        return true;
                    } else {
                        checkState(ignoreMissedDays.get(), ERROR, nextDay);
                    }
                    curDate = curDate.plusDays(1);
                }
                return false;
            }

            @Override
            public DayLog next() {

                while (curDate.isBefore(finishDate)) {
                    curDate = curDate.plusDays(1);
                    YtTable logTable = new YtTable(logPathProvider.apply(curDate));
                    if (checkTableExistence(ytOperator, logTable)) {
                        return new DayLog(curDate, logTable);
                    } else {
                        checkState(ignoreMissedDays.get(), ERROR, curDate);
                    }
                }
                throw new IllegalStateException(String.format(ERROR, curDate));
            }
        };
    }

    private boolean checkTableExistence(YtOperator ytOperator, YtTable logTable) {
        return ytOperator.exists(logTable) &&
                ytOperator.readTableNumericAttribute(logTable, ROW_COUNT_ATTRIBUTE_NAME) > 0;
    }

    public enum LastDayMode {

        /**
         * Самым свежим считается последний доступный подневный лог
         */
        USE_LAST_DAY,

        /**
         * Самым свежим считается предпоследний доступный подневный лог.
         * Нужно, чтобы не начать читать неполный лог.
         */
        SKIP_LAST_DAY,
    }
}
