package ru.yandex.direct.jobs.segment.log;

import java.math.BigInteger;
import java.time.LocalDate;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.audience.client.model.SegmentContentType;
import ru.yandex.direct.jobs.segment.common.meta.SegmentKey;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static org.apache.commons.collections.CollectionUtils.isEmpty;

public class SegmentSourceData {

    private final Set<SegmentKey> fetchedSegmentsKeys;
    private final Map<SegmentKey, Set<BigInteger>> segmentKeyToUids;
    private final LocalDate lastReadLogDate;
    private final LocalDate mostFreshLogDate;
    private final SegmentContentType contentType;

    /**
     * @param fetchedSegmentsKeys идентификаторы сегментов, по которым делалась выборка данных из логов
     * @param segmentKeyToUids    данные сегментов
     * @param lastReadLogDate     дата, до которой были прочитаны данные сегментов
     * @param mostFreshLogDate    самая свежая дата, за которую есть логи
     */
    private SegmentSourceData(Collection<SegmentKey> fetchedSegmentsKeys,
                              Map<SegmentKey, Set<BigInteger>> segmentKeyToUids,
                              LocalDate lastReadLogDate,
                              LocalDate mostFreshLogDate,
                              SegmentContentType contentType) {
        checkNotNull(fetchedSegmentsKeys, "fetchedSegmentsKeys");
        checkNotNull(segmentKeyToUids, "segmentKeyToUids");
        checkArgument(lastReadLogDate == null ^ !fetchedSegmentsKeys.isEmpty(),
                "lastReadLogDate and fetchedSegmentsKeys must be both defined or not");
        this.fetchedSegmentsKeys = ImmutableSet.copyOf(fetchedSegmentsKeys);
        this.segmentKeyToUids = ImmutableMap.copyOf(segmentKeyToUids);
        this.lastReadLogDate = lastReadLogDate;
        this.mostFreshLogDate = mostFreshLogDate;
        this.contentType = contentType;
    }

    public static SegmentSourceData sourceData(Collection<SegmentKey> fetchedSegmentsKeys,
                                               Map<SegmentKey, Set<BigInteger>> segmentKeyToUids,
                                               LocalDate lastReadLogDate,
                                               LocalDate mostFreshLogDate,
                                               SegmentContentType contentType) {
        return new SegmentSourceData(fetchedSegmentsKeys, segmentKeyToUids,
                lastReadLogDate, mostFreshLogDate, contentType);
    }

    public static SegmentSourceData noSourceData(LocalDate mostFreshLogDate) {
        return new SegmentSourceData(emptyList(), emptyMap(), null, mostFreshLogDate, SegmentContentType.YUID);
    }

    public Set<SegmentKey> getFetchedSegmentsKeys() {
        return fetchedSegmentsKeys;
    }

    public Map<SegmentKey, Set<BigInteger>> getSegmentKeyToUids() {
        return segmentKeyToUids;
    }

    public Set<BigInteger> getUids(SegmentKey segmentKey) {
        return segmentKeyToUids.get(segmentKey);
    }

    public Set<BigInteger> getUidsOrEmptySet(SegmentKey segmentKey) {
        return segmentKeyToUids.getOrDefault(segmentKey, new HashSet<>());
    }

    public boolean wereLogsRead() {
        return !isEmpty(fetchedSegmentsKeys);
    }

    @Nullable
    public LocalDate getLastReadLogDate() {
        return lastReadLogDate;
    }

    @Nullable
    public LocalDate getMostFreshLogDate() {
        return mostFreshLogDate;
    }

    public SegmentContentType getContentType() {
        return contentType;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SegmentSourceData that = (SegmentSourceData) o;
        return Objects.equals(fetchedSegmentsKeys, that.fetchedSegmentsKeys) &&
                Objects.equals(segmentKeyToUids, that.segmentKeyToUids) &&
                Objects.equals(lastReadLogDate, that.lastReadLogDate) &&
                Objects.equals(mostFreshLogDate, that.mostFreshLogDate) &&
                contentType == that.contentType;
    }

    @Override
    public int hashCode() {
        return Objects.hash(fetchedSegmentsKeys, segmentKeyToUids, lastReadLogDate, mostFreshLogDate, contentType);
    }

    @Override
    public String toString() {
        return "SegmentSourceData{" +
                "fetchedSegmentsKeys=" + fetchedSegmentsKeys +
                ", segmentKeyToUids=" + segmentKeyToUids +
                ", lastReadLogDate=" + lastReadLogDate +
                ", mostFreshLogDate=" + mostFreshLogDate +
                ", contentType=" + contentType +
                '}';
    }
}
