package ru.yandex.direct.jobs.segment.log.common

import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Repository
import ru.yandex.direct.audience.client.model.SegmentContentType
import ru.yandex.direct.core.entity.adgroup.model.AdShowType
import ru.yandex.direct.jobs.segment.common.meta.SegmentKey
import ru.yandex.direct.jobs.segment.log.BaseIntermediateSegmentYtRepository
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtField
import java.math.BigInteger
import java.sql.ResultSet
import java.time.LocalDate

/**
 * Репозиторий для работы с таблицей
 * https://yt.yandex-team.ru/hahn/navigation?path=//home/direct/common/segments/1d
 */
@Repository
class CommonIntermediateSegmentYtRepository(
    ytProvider: YtProvider,
    @Value("\${segment.common.intermediate_path}") intermediateTablePath: String,
): BaseIntermediateSegmentYtRepository(
    ytProvider,
    intermediateTablePath
) {
    override fun getContentType() = SegmentContentType.CRYPTA_ID

    override fun getCountQuery(segmentKeys: Collection<SegmentKey>, from: LocalDate, to: LocalDate): String {
        return buildCountQuery(intermediateTableRootPath, segmentKeys, from, to)
    }

    override fun getDataQuery(segmentKeys: Collection<SegmentKey>, from: LocalDate, to: LocalDate): String {
        return buildDataQuery(intermediateTableRootPath, segmentKeys, from, to)
    }

    override fun mapCountResult(resultSet: ResultSet, result: MutableMap<SegmentKey, Long>): Any? {
        val segmentKey = extractSegmentKeyFromResultSet(resultSet)
        val count = getCount(resultSet.getObject(2))
        result[segmentKey] = count
        return null
    }

    override fun mapDataResult(resultSet: ResultSet, result: MutableMap<SegmentKey, MutableSet<BigInteger>>): Any? {
        val segmentKey = extractSegmentKeyFromResultSet(resultSet)
        val uid = BigInteger(resultSet.getString(2))
        result.computeIfAbsent(segmentKey) { mutableSetOf() }.add(uid)
        return null
    }

    private fun extractSegmentKeyFromResultSet(resultSet: ResultSet): SegmentKey {
        val adGroupId = resultSet.getLong(GROUPEXPORTID.name)
        return SegmentKey(adGroupId, AdShowType.START)
    }

    companion object {
        private val GROUPEXPORTID = YtField("groupexportid", Long::class.java)

        private const val SQL_TABLE_LINE = "    AsStruct(%s as groupexportid)"

        private val SQL_COUNT_TEMPL = """
            ${"$"}data = AsList(
            %s
            );
            
            select t.groupexportid as groupexportid, count(*)
            from range(`%s`, `%s`, `%s`) as t
            join as_table(${"$"}data) as d on t.groupexportid = d.groupexportid
            group by t.groupexportid;
            """.trimIndent()

        private val SQL_DATA_TEMPL = """
            ${"$"}data = AsList(
            %s
            );
            
            select t.groupexportid as groupexportid, cryptaidv2
            from range(`%s`, `%s`, `%s`) as t
            join as_table(${"$"}data) as d on t.groupexportid = d.groupexportid;
            """.trimIndent()

        @JvmStatic
        fun buildCountQuery(
            intermediateTableRootPath: String,
            segmentKeys: Collection<SegmentKey>,
            from: LocalDate,
            to: LocalDate
        ) = String.format(SQL_COUNT_TEMPL, buildMergedTable(segmentKeys), intermediateTableRootPath, from, to)

        @JvmStatic
        fun buildDataQuery(
            intermediateTableRootPath: String,
            segmentKeys: Collection<SegmentKey>,
            from: LocalDate,
            to: LocalDate
        ) = String.format(SQL_DATA_TEMPL, buildMergedTable(segmentKeys), intermediateTableRootPath, from, to)

        private fun buildMergedTable(segmentKeys: Collection<SegmentKey>) =
            segmentKeys.joinToString(",\n") { String.format(SQL_TABLE_LINE, it.adGroupId) }
    }
}
