package ru.yandex.direct.jobs.segment.log.cpmdefault

import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Repository
import ru.yandex.direct.audience.client.model.SegmentContentType
import ru.yandex.direct.core.entity.adgroup.model.AdShowType
import ru.yandex.direct.jobs.segment.common.meta.SegmentKey
import ru.yandex.direct.jobs.segment.log.BaseIntermediateSegmentYtRepository
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtField
import java.math.BigInteger
import java.sql.ResultSet
import java.time.LocalDate
import java.util.Locale

/**
 * Репозиторий для работы с таблицей
 * https://yt.yandex-team.ru/hahn/navigation?path=//home/direct/segments/1d
 */
@Repository
class CpmDefaultIntermediateSegmentYtRepository(
    ytProvider: YtProvider,
    @Value("\${segment.cpm_default.intermediate_path}") intermediateTablePath: String
) : BaseIntermediateSegmentYtRepository(
    ytProvider,
    intermediateTablePath
) {
    override fun getContentType() = SegmentContentType.YUID

    override fun getCountQuery(segmentKeys: Collection<SegmentKey>, from: LocalDate, to: LocalDate): String {
        return buildCountQuery(intermediateTableRootPath, segmentKeys, from, to)
    }

    override fun getDataQuery(segmentKeys: Collection<SegmentKey>, from: LocalDate, to: LocalDate): String {
        return buildDataQuery(intermediateTableRootPath, segmentKeys, from, to)
    }

    override fun mapCountResult(resultSet: ResultSet, result: MutableMap<SegmentKey, Long>): Any? {
        val segmentKey = extractSegmentKeyFromResultSet(resultSet)
        val count = getCount(resultSet.getObject(3))
        result[segmentKey] = count
        return null
    }

    override fun mapDataResult(resultSet: ResultSet, result: MutableMap<SegmentKey, MutableSet<BigInteger>>): Any? {
        val segmentKey = extractSegmentKeyFromResultSet(resultSet)
        val uid = BigInteger(resultSet.getString(3))
        result.computeIfAbsent(segmentKey) { mutableSetOf() }.add(uid)
        return null
    }

    private fun extractSegmentKeyFromResultSet(resultSet: ResultSet): SegmentKey {
        val adGroupId = resultSet.getString(GROUPEXPORTID.name).toLong()
        val type = getAdShowTypeFromString(resultSet.getString(ACTION.name))
        return SegmentKey(adGroupId, type)
    }

    /**
     * Преобразование типа просмотра из формата YT во внутренний формат
     */
    private fun getAdShowTypeFromString(str: String): AdShowType {
        val toUpper = str.uppercase(Locale.getDefault()).replace('-', '_')
        return AdShowType.valueOf(toUpper)
    }

    companion object {
        private val GROUPEXPORTID = YtField("groupexportid", String::class.java)
        private val ACTION = YtField("action", String::class.java)

        private const val SQL_TABLE_LINE = "    AsStruct('%s' as groupexportid, '%s' as action)"

        private val SQL_COUNT_TEMPL = """
            ${"$"}data = AsList(
            %s
            );
            
            select t.groupexportid as groupexportid, t.action as action, count(*)
            from range(`%s`, `%s`, `%s`) as t
            join as_table(${"$"}data) as d on t.groupexportid = d.groupexportid and t.action = d.action
            group by t.groupexportid, t.action;
            """.trimIndent()

        private val SQL_DATA_TEMPL = """
            ${"$"}data = AsList(
            %s
            );
            
            select t.groupexportid as groupexportid, t.action as action, uniqid
            from range(`%s`, `%s`, `%s`) as t
            join as_table(${"$"}data) as d on t.groupexportid = d.groupexportid and t.action = d.action;
            """.trimIndent()

        @JvmStatic
        fun buildCountQuery(
            intermediateTableRootPath: String,
            segmentKeys: Collection<SegmentKey>,
            from: LocalDate,
            to: LocalDate
        ) = String.format(SQL_COUNT_TEMPL, buildMergedTable(segmentKeys), intermediateTableRootPath, from, to)

        @JvmStatic
        fun buildDataQuery(
            intermediateTableRootPath: String,
            segmentKeys: Collection<SegmentKey>,
            from: LocalDate,
            to: LocalDate
        ) = String.format(SQL_DATA_TEMPL, buildMergedTable(segmentKeys), intermediateTableRootPath, from, to)

        private fun buildMergedTable(segmentKeys: Collection<SegmentKey>) =
            segmentKeys.joinToString(",\n") { buildTableLine(it) }

        private fun buildTableLine(segmentKey: SegmentKey) =
            String.format(SQL_TABLE_LINE, segmentKey.adGroupId, segmentTypeToAction(segmentKey.segmentType))

        private fun segmentTypeToAction(segmentType: AdShowType) =
            segmentType.name.lowercase(Locale.getDefault()).replace("_", "-")
    }
}
