package ru.yandex.direct.jobs.segment.log.greedy;

import java.math.BigInteger;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import com.google.gson.Gson;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.audience.client.model.SegmentContentType;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.adgroup.model.UsersSegment;
import ru.yandex.direct.jobs.base.logdatatransfer.LogFetchingStrategy;
import ru.yandex.direct.jobs.segment.common.meta.SegmentKey;
import ru.yandex.direct.jobs.segment.log.IntermediateSegmentYtRepository;
import ru.yandex.direct.jobs.segment.log.SegmentSourceData;

import static java.util.Comparator.comparing;
import static java.util.Comparator.naturalOrder;
import static java.util.Comparator.reverseOrder;
import static java.util.Map.Entry.comparingByValue;
import static ru.yandex.direct.jobs.segment.common.SegmentUtils.segmentKeyExtractor;
import static ru.yandex.direct.jobs.segment.log.SegmentSourceData.noSourceData;
import static ru.yandex.direct.jobs.segment.log.SegmentSourceData.sourceData;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Рассмотрим на примере, как работает стратегия.
 * <p>
 * На вход поступают несколько сегментов с разными датами последнего обновления
 * (если точнее, то с датами, за которые были прочитаны логи и загружены в сегменты Аудиторий).
 * <p>
 * id/дата      1.01    2.01    3.01    4.01    5.01
 * <p>
 * 1             +
 * 2             +
 * 3                     +
 * 4                     +
 * 5                             +
 * <p>
 * Сегменты 1 и 2 были обновлены из логов 1 января. Соответственно, они не обновлялись дольше всех.
 * <p>
 * 1. За один вызов метода fetch могут быть обработаны только сегменты
 * с самой старой датой обновления, то есть сегменты 1 и 2.
 * 2. Для этих сегментов делается запрос за количеством записей в логах с (дата последнего обновления + 1)
 * до даты самых свежих логов.
 * 3. С учетом количества записей выбирается такое подмножество сегментов, что суммарное количество их данных
 * не превышает установленный лимит. Для выбора подмножества применяется простой жадный алгоритм с сортировкой.
 * 4. Если ни один сегмент не имеет размер данных меньше лимита (не удаётся собрать пачку),
 * тогда алгоритм возвращается к пункту 2, при этом период выборки сокращается вдвое.
 * 5. Когда подмножество сегментов выбрано, делается запрос за самими данными сегментов.
 */
public class GreedySegmentLogFetchingStrategy implements LogFetchingStrategy<UsersSegment, SegmentSourceData> {

    private static final Logger logger = LoggerFactory.getLogger(GreedySegmentLogFetchingStrategy.class);

    private final IntermediateSegmentYtRepository intermediateSegmentYtRepository;
    private final Supplier<Long> packLimitSupplier;
    private Long packLimit;

    public GreedySegmentLogFetchingStrategy(IntermediateSegmentYtRepository intermediateSegmentYtRepository,
                                            Supplier<Long> packLimitSupplier) {
        this.intermediateSegmentYtRepository = intermediateSegmentYtRepository;
        this.packLimitSupplier = packLimitSupplier;
    }

    public GreedySegmentLogFetchingStrategy(IntermediateSegmentYtRepository intermediateSegmentYtRepository,
                                            PpcProperty<Long> packLimitProperty,
                                            long defaultPackLimit) {
        this.intermediateSegmentYtRepository = intermediateSegmentYtRepository;
        this.packLimitSupplier = () -> packLimitProperty.getOrDefault(defaultPackLimit);
    }

    @Override
    public SegmentSourceData fetch(List<UsersSegment> segmentMetaList) {
        initPackLimit();

        logger.info("getting the oldest log date...");
        LocalDate oldestLogDate = intermediateSegmentYtRepository.getOldestLogDate();
        logger.info("oldest log date: {}", oldestLogDate);

        logger.info("getting the most fresh log date...");
        LocalDate mostFreshLogDate = intermediateSegmentYtRepository.getMostFreshLogDate();
        logger.info("most fresh log date: " + mostFreshLogDate);

        if (segmentMetaList.isEmpty()) {
            logger.info("no segment meta for fetching data, return...");
            return noSourceData(mostFreshLogDate);
        }

        LocalDate minUpdateDate = getMinUpdateDate(segmentMetaList);
        logger.info("minimal last read log date of fetched segments: {}", minUpdateDate);

        // минимальная дата обновления может быть позже самых свежих логов,
        // когда клиент только что выставил галочку "собирать сегмент" -
        // это новосозданная meta (запись в ppc.video_segment_goals)
        if (minUpdateDate.equals(mostFreshLogDate) || minUpdateDate.isAfter(mostFreshLogDate)) {
            logger.info("minimal update date of segments ({}) is equal to or after the most fresh log date ({}): " +
                    "no segment meta for fetching data, return...", minUpdateDate, mostFreshLogDate);
            return noSourceData(mostFreshLogDate);
        }

        logger.info("creating optimized pack of segments to fetch...");
        Pack packToFetch = getPack(segmentMetaList, oldestLogDate, mostFreshLogDate, minUpdateDate);

        logger.info("fetching pack of segments...");
        Map<SegmentKey, Set<BigInteger>> segmentData = intermediateSegmentYtRepository
                .getData(packToFetch.segmentKeys, packToFetch.dateFrom, packToFetch.dateTo);
        long totalUids = StreamEx.of(segmentData.values())
                .mapToInt(Set::size)
                .sum();
        logger.info("pack of segments fetched (segments: {}, total unique uids: {})", segmentData.size(), totalUids);

        if (packToFetch.segmentKeys.size() != segmentData.size()) {
            logger.warn("fetched less segments ({}) than planned ({})",
                    segmentData.size(), packToFetch.segmentKeys.size());
        }

        SegmentContentType contentType = intermediateSegmentYtRepository.getContentType();
        return sourceData(packToFetch.segmentKeys, segmentData, packToFetch.dateTo,
                packToFetch.mostFreshLogDate, contentType);
    }

    private void initPackLimit() {
        if (packLimit == null) {
            packLimit = packLimitSupplier.get();
        }
    }

    private LocalDate getMinUpdateDate(List<UsersSegment> segmentMetaList) {
        return StreamEx.of(segmentMetaList)
                .min(comparing(this::extractUpdateDate, naturalOrder()))
                .map(this::extractUpdateDate)
                .orElseThrow(() -> new IllegalArgumentException("this never can happen: " +
                        "may be empty list was passes as an argument"));
    }

    private Pack getPack(List<UsersSegment> segmentMetaList,
                         LocalDate oldestLogDate,
                         LocalDate mostFreshLogDate,
                         LocalDate minUpdateDate) {

        LocalDate dateFrom = minUpdateDate.plusDays(1);
        if (dateFrom.isBefore(oldestLogDate)) {
            dateFrom = oldestLogDate;
            logger.info("min last read log date ({}) is too old, start log date is set to {}",
                    minUpdateDate, oldestLogDate);
        }

        List<UsersSegment> mostStaleSegments = getSegmentsUpdatedTill(segmentMetaList, dateFrom);
        List<SegmentKey> segmentKeys = mapList(mostStaleSegments, segmentKeyExtractor());

        logger.info("potential segments to read from {}: {}", dateFrom, mostStaleSegments.size());

        Map<SegmentKey, Long> segmentKeyToCountMap;
        LocalDate dateTo = null;
        Map<SegmentKey, Long> segmentsToFetch;
        do {
            if (dateTo == null) { // первая итерация
                dateTo = mostFreshLogDate;
                logger.info("first iteration: try to create pack of segments from {} to {}",
                        dateFrom, dateTo);
            } else { // последующие итерации
                dateTo = getHalfDate(dateFrom, dateTo);
                logger.info("too many logs to read entire period, try to create pack of segments from {} to {}",
                        dateFrom, dateTo);
            }

            segmentKeyToCountMap = intermediateSegmentYtRepository.getCount(segmentKeys, dateFrom, dateTo);
            for (SegmentKey sk : segmentKeys) {
                segmentKeyToCountMap.putIfAbsent(sk, 0L);
            }
            segmentsToFetch = getPack(segmentKeyToCountMap);

        } while (segmentsToFetch.isEmpty() && dateTo.isAfter(dateFrom));

        if (segmentsToFetch.isEmpty()) {
            logger.error("can't create pack with current limit (limit: {}, from: {}, to: {}, count: {})",
                    packLimit, dateFrom, dateTo, new Gson().toJson(segmentKeyToCountMap));
            throw new IllegalStateException("can't create pack with current limit, see error log for details");
        }

        long totalUids = StreamEx.of(segmentsToFetch.values())
                .mapToLong(count -> count)
                .sum();
        logger.info("segments pack created (segments: {}, total uids (not unique): {}, dateFrom: {}, dateTo: {})",
                segmentsToFetch.size(), totalUids, dateFrom, dateTo);
        logger.info("segments pack details: {}", segmentsToFetch);

        return new Pack(segmentsToFetch.keySet(), dateFrom, dateTo, mostFreshLogDate);
    }

    private List<UsersSegment> getSegmentsUpdatedTill(List<UsersSegment> segmentMetaList, LocalDate date) {
        return filterList(segmentMetaList,
                segmentMeta -> {
                    LocalDate segmentDate = extractUpdateDate(segmentMeta);
                    return segmentDate.isBefore(date);
                });
    }

    private LocalDate extractUpdateDate(UsersSegment segmentMeta) {
        return segmentMeta.getLastSuccessUpdateTime().toLocalDate();
    }

    /**
     * Возвращает список id сегментов, данные по которым можно за раз загрузить в память из YT и обработать.
     * Использует простой жадный алгоритм.
     */
    private Map<SegmentKey, Long> getPack(Map<SegmentKey, Long> segmentKeyToCountMap) {
        List<Map.Entry<SegmentKey, Long>> sortedEntries = new ArrayList<>(segmentKeyToCountMap.entrySet());
        sortedEntries.sort(comparingByValue(reverseOrder()));

        long curSize = 0;
        Map<SegmentKey, Long> pack = new HashMap<>();
        for (Map.Entry<SegmentKey, Long> entry : sortedEntries) {
            SegmentKey segmentKey = entry.getKey();
            Long segmentSize = entry.getValue();
            if (curSize + segmentSize <= packLimit) {
                pack.put(segmentKey, segmentSize);
                curSize += segmentSize;
            }
        }
        return pack;
    }

    private LocalDate getHalfDate(LocalDate from, LocalDate to) {
        long daysBetween = ChronoUnit.DAYS.between(from, to);
        long halfDays = daysBetween / 2;
        return halfDays == 0 ? from : from.plusDays(halfDays);
    }

    private static class Pack {
        Set<SegmentKey> segmentKeys;
        LocalDate dateFrom;
        LocalDate dateTo;
        LocalDate mostFreshLogDate;

        public Pack(Set<SegmentKey> segmentKeys,
                    LocalDate dateFrom,
                    LocalDate dateTo,
                    LocalDate mostFreshLogDate) {
            this.segmentKeys = segmentKeys;
            this.dateFrom = dateFrom;
            this.dateTo = dateTo;
            this.mostFreshLogDate = mostFreshLogDate;
        }
    }
}
