package ru.yandex.direct.jobs.sessionvariables;

import java.time.LocalDateTime;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.sessionvariables.repository.StoredVarsRepository;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Очищает таблицу {@link ru.yandex.direct.dbschema.monitor.tables.StoredVars}
 * от значений старше {@code MAX_AGE} = {@value MAX_AGE} дней
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 2),
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@ParametersAreNonnullByDefault
@Hourglass(cronExpression = "0 30 1-8 * * ?", needSchedule = TypicalEnvironment.class)
public class StoredVarsCleaner extends DirectJob {
    private static final int MAX_AGE = 7;

    private static final Logger logger = LoggerFactory.getLogger(StoredVarsCleaner.class);

    private final StoredVarsRepository storedVarsRepository;

    @Autowired
    public StoredVarsCleaner(StoredVarsRepository storedVarsRepository) {
        this.storedVarsRepository = storedVarsRepository;
    }

    @Override
    public void execute() {
        LocalDateTime borderDateTime = LocalDateTime.now().minusDays(MAX_AGE);

        logger.info("delete outdated records from border time {}", borderDateTime);
        int deletedRows = storedVarsRepository.deleteStoredVarsOlderThan(borderDateTime);
        logger.info("deleted {} records", deletedRows);
    }
}
