package ru.yandex.direct.jobs.shoprating;

import java.net.IDN;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.net.InternetDomainName;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableStringRequest;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.utils.JsonUtils;

import static org.apache.commons.lang3.StringUtils.isNotEmpty;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Обновление рейтингов магазинов Маркета
 * Джоба заполняет таблицы ppcdict.domains_dict и ppcdict.market_ratings
 * Доменам, для которых нет рейтинга в полученном из Маркета файле, присваивается рейтинг -1
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3),
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION}
)
@Hourglass(periodInSeconds = 3600, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class JobUpdateShopRating extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(JobUpdateShopRating.class);

    private final DomainService domainService;
    private final HostingsHandler hostingsHandler;
    private final SolomonPushClient solomonPushClient;
    private final String url;
    private final int minimumRatingsCount;
    private final ParallelFetcherFactory parallelFetcherFactory;

    @Autowired
    public JobUpdateShopRating(DomainService domainService, AsyncHttpClient asyncHttpClient,
                               HostingsHandler hostingsHandler, SolomonPushClient solomonPushClient,
                               @Value("${shop_rating_url}") String url,
                               @Value("${minimum_shop_ratings_count}") int minimumRatingsCount) {
        this.domainService = domainService;
        this.hostingsHandler = hostingsHandler;
        this.solomonPushClient = solomonPushClient;
        this.url = url;
        this.minimumRatingsCount = minimumRatingsCount;

        parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings());
    }

    @Override
    public void execute() {
        String ratingInfo = downloadRatingInfo(url);

        Map<String, Long> newRatings = parseRatingInfo(ratingInfo);

        if (newRatings.size() < minimumRatingsCount) {
            throw new RuntimeException(String.format("Something wrong with shops ratings. Expected more than %d "
                    + "ratings, got %d ratings", minimumRatingsCount, newRatings.size()));
        }

        logger.info("Got {} ratings from {}", newRatings.size(), url);

        int updated = domainService.updateMarketDomainRatings(newRatings);

        logger.info("Updated {} domains", updated);

        sendMetrics(newRatings, updated);
    }

    String downloadRatingInfo(String url) {
        logger.info("Start downloading shops ratings from {}", url);

        try (ParallelFetcher<String> fetcher = parallelFetcherFactory.getParallelFetcher()) {
            RequestBuilder builder = new RequestBuilder().setUrl(url);
            return fetcher.executeWithErrorsProcessing(new ParsableStringRequest(0, builder.build())).getSuccess();
        }
    }

    Map<String, Long> parseRatingInfo(String ratingInfo) {
        Map<String, Long> ratings = new HashMap<>();
        for (ShopRating shopRating : JsonUtils.fromJson(ratingInfo, new TypeReference<List<ShopRating>>() {
        })) {
            String domain;
            Long rating = shopRating.getRating();

            try {
                domain = IDN.toASCII(hostingsHandler.stripWww(shopRating.getDomain()));
            } catch (IllegalArgumentException ignored) {
                // возможно какая-то произвольная длинная строка, пропустим такую ниже
                domain = null;
            }

            if (isNotEmpty(domain) && validateDomain(domain)) {
                ratings.put(domain, rating);
            } else {
                logger.info("Bad domain found: {}", shopRating.getDomain());
            }
        }
        return ratings;
    }

    boolean validateDomain(String domain) {
        try {
            return InternetDomainName.from(domain).parts().size() > 1;
        } catch (IllegalArgumentException e) {
            return false;
        }
    }

    private void sendMetrics(Map<String, Long> ratings, int updated) {
        var registry = SolomonUtils.newPushRegistry("flow", "JobUpdateShopRating");
        registry.gaugeInt64("domains_from_market_count").set(ratings.size());
        registry.gaugeInt64("domains_changed_rating_count").set(updated);
        solomonPushClient.sendMetrics(registry);
    }
}
