package ru.yandex.direct.jobs.sitelinks

import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.jobs.sitelinks.ImportSitelinksUtils.getCurrentPath
import ru.yandex.direct.jobs.sitelinks.ImportSitelinksUtils.getLinkTargetPath
import ru.yandex.direct.jobs.sitelinks.ImportSitelinksUtils.getWorkDirectory
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.inside.yt.kosher.cypress.YPath

/**
 * Удаляем старые таблицы (сортировка по имени таблицы), оставляя последние `config.tablesToKeep`.
 * Также, никогда не удаляем таблицу, на которую ссылается симлинк `current`
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(hours = 12, minutes = 20),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_2],
    notifications = [OnChangeNotification(
        recipient = [
            NotificationRecipient.CHAT_API_MONITORING,
            NotificationRecipient.LOGIN_A_DUBOV,
        ],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 6 * 60 * 60, needSchedule = ProductionOnly::class)
class ImportSitelinksCleanupJob(
    private val ytProvider: YtProvider,
    directConfig: DirectConfig,
) : DirectJob() {

    private val logger = LoggerFactory.getLogger(ImportSitelinksFromBnoJob::class.java)

    private val config = ImportSitelinksConfig(directConfig)

    override fun execute() {
        config.outputClusters.forEach { cluster ->
            cleanupOnCluster(cluster)
        }
    }

    private fun cleanupOnCluster(cluster: YtCluster) {
        val ytOperator = ytProvider.getOperator(cluster)
        val directory = getWorkDirectory(ytProvider, cluster, config.directory)

        val tables = ytOperator.yt.cypress().list(YPath.simple(directory))
            .filter { it.isStringNode }
            .map { it.stringValue() }
            .filter { table -> table.startsWith(ImportSitelinksUtils.tableNamePrefix) }
            .sorted()
        logger.info("All tables: {}", tables)

        val tablesToKeep = tables.takeLast(config.tablesToKeep).toSet()
        logger.info("Last tables to keep: {}", tablesToKeep)

        val currentSymlink = getLinkTargetPath(ytOperator, getCurrentPath(directory))
        val currentSymlinkTable = currentSymlink.name()
        logger.info("Current symlink is {}", currentSymlinkTable)

        val tablesToRemove = tables - tablesToKeep - currentSymlinkTable
        logger.info("Going to remove tables {}", tablesToRemove)

        tablesToRemove.forEach { table ->
            val tablePath = YPath.simple(directory).child(table)
            logger.info("Removing table {}", tablePath)
            ytOperator.yt.cypress().remove(tablePath)
        }
    }
}
