package ru.yandex.direct.jobs.sitelinks

import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.jobs.sitelinks.ImportSitelinksUtils.getCreatedAt
import ru.yandex.direct.jobs.sitelinks.ImportSitelinksUtils.getCurrentPath
import ru.yandex.direct.jobs.sitelinks.ImportSitelinksUtils.getLinkTargetPath
import ru.yandex.direct.jobs.sitelinks.ImportSitelinksUtils.getTableName
import ru.yandex.direct.jobs.sitelinks.ImportSitelinksUtils.getWorkDirectory
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.liveresource.LiveResourceFactory
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.transfermanagerutils.TransferManager
import ru.yandex.direct.transfermanagerutils.TransferManagerConfig
import ru.yandex.direct.transfermanagerutils.TransferManagerJobConfig
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.ytwrapper.YtPathUtil
import ru.yandex.direct.ytwrapper.YtUtils.isClusterAvailable
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtOperator
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.inside.yt.kosher.common.DataSize
import ru.yandex.inside.yt.kosher.cypress.YPath
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree
import ru.yandex.inside.yt.kosher.operations.specs.JobIo
import ru.yandex.inside.yt.kosher.operations.specs.MergeMode
import ru.yandex.inside.yt.kosher.operations.specs.MergeSpec
import ru.yandex.inside.yt.kosher.tables.TableWriterOptions
import java.time.Duration
import java.time.Instant
import kotlin.math.min

/**
 * Джоба по импорту данных сайтлинков
 *
 * 1. С помощью YQL запроса получаем сгруппированные сайтлинки на arnold
 * 2. Отдельной merge операцией изменяем размер блоков таблицы
 * 3. Через transfer manager копируем на seneca-*
 * 4. Делаем таблицу динамической и монтируем
 * 5. Меняем линк /current на новую таблицу
 * 6. Размонтируем старую таблицу
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(hours = 8),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_2],
    notifications = [OnChangeNotification(
        recipient = [
            NotificationRecipient.CHAT_API_MONITORING,
            NotificationRecipient.LOGIN_A_DUBOV,
        ],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(cronExpression = "0 0 */6 * * ?", needSchedule = ProductionOnly::class)
class ImportSitelinksFromBnoJob(
    private val ppcPropertiesSupport: PpcPropertiesSupport,
    private val ytProvider: YtProvider,
    transferManagerConfig: TransferManagerConfig,
    directConfig: DirectConfig
) : DirectJob() {

    private val logger = LoggerFactory.getLogger(ImportSitelinksFromBnoJob::class.java)

    private val mergeAwaitTimeout = Duration.ofMinutes(5)
    private val transferAwaitTimeout = Duration.ofMinutes(10)
    private val mountTimeout = Duration.ofSeconds(30)

    private val yqlQuery = LiveResourceFactory.get("classpath:///sitelinks/importSitelinksFromBno.sql").content

    private val config = ImportSitelinksConfig(directConfig)

    private val transferManager = TransferManager(transferManagerConfig)

    private val isJobEnabled
        get() = ppcPropertiesSupport
            .get(PpcPropertyNames.IMPORT_SITELINKS_FROM_BNO_ENABLED)
            .getOrDefault(false)

    override fun execute() {
        if (!isJobEnabled) {
            logger.info("Job is disabled")
            return
        }

        try {
            process()
        } catch (e: RuntimeException) {
            logger.error("Job run failed", e)
        } finally {
            updateJugglerStatus()
        }
    }

    private fun updateJugglerStatus() {
        logger.info("Calculating table freshness")
        val clusterFreshness: List<Pair<YtCluster, Instant>> = try {
            config.outputClusters
                .filter { cluster -> isClusterAvailable(ytProvider, cluster) }
                .associateWith { cluster -> getCurrentPath(getWorkDirectory(ytProvider, cluster, config.directory)) }
                .mapValues { (cluster, path) -> getCreatedAt(ytProvider, cluster, YPath.simple(path)) }
                .filterValues { it != null }
                .mapValues { (_, createdAt) -> createdAt as Instant }
                .toList()
                .sortedByDescending { (_, createdAt) -> createdAt }
        } catch (e: RuntimeException) {
            logger.info("Failed to calculate output freshness", e)
            setJugglerStatus(JugglerStatus.CRIT, "Failed to calculate output freshness")
            return
        }

        logger.info("Cluster freshness: {}", clusterFreshness)

        // берем второй по свежести кластер, чтобы проверять что есть "хотя бы два" актуальных
        val count = min(2, config.outputClusters.size)

        if (clusterFreshness.size < count) {
            val message = "Could not find enough ($count) output tables"
            logger.info(message)
            setJugglerStatus(JugglerStatus.CRIT, message)
            return
        }

        val (cluster, freshness) = clusterFreshness[count - 1]
        if (Duration.between(freshness, Instant.now()) > config.freshDuration) {
            val message = "Output tables are too old. The second fresh one is created at $freshness on cluster $cluster"
            logger.info(message)
            setJugglerStatus(JugglerStatus.CRIT, message)
        }
    }

    private fun process() {
        val importCluster = getImportCluster()
        if (importCluster == null) {
            logger.error("No cluster with input table available")
            return
        }

        val tableName = getTableName()
        val outputTableByCluster = config.outputClusters
            .filter { isClusterAvailable(ytProvider, it) }
            .associateWith { cluster ->
                val clusterWorkDirectory = getWorkDirectory(ytProvider, cluster, config.directory)
                YPath.simple(clusterWorkDirectory).child(tableName)
            }


        if (outputTableByCluster.isEmpty()) {
            logger.error("No output clusters available")
            return
        }

        val ytOperator = ytProvider.getOperator(importCluster)

        val yqlResultTable: YPath = runYql(ytOperator)

        val tableToTransfer: YPath = try {
            fixTableBlockSize(ytOperator, yqlResultTable)
        } finally {
            logger.info("Removing temporary table $yqlResultTable on cluster ${importCluster.getName()}")
            ytOperator.yt.cypress().remove(yqlResultTable)
        }

        val transferredClusters: List<YtCluster> = try {
            transferTable(importCluster, tableToTransfer, outputTableByCluster)
        } finally {
            logger.info("Removing temporary table $tableToTransfer on cluster ${importCluster.getName()}")
            ytOperator.yt.cypress().remove(tableToTransfer)
        }

        if (transferredClusters.isEmpty()) {
            logger.error("Transfer failed for all target clusters")
            return
        }

        transferredClusters
            .associateWith { cluster -> outputTableByCluster[cluster]!! }
            .forEach { (cluster, table) ->
                try {
                    processOutput(cluster, table)
                } catch (e: RuntimeException) {
                    logger.error("Failed to process table on cluster ${cluster.getName()}", e)
                }
            }
    }

    private fun runYql(ytOperator: YtOperator): YPath {
        val yqlResultTable = YtPathUtil.generateTemporaryPath()

        logger.info("Executing query on cluster ${ytOperator.cluster.getName()} with output $yqlResultTable")
        ytOperator.yqlExecute(yqlQuery, config.importTablePath, yqlResultTable.toString())

        return YPath.simple(yqlResultTable)
    }

    /**
     * Изменяем размер блоков таблицы, перед конвертацией в динамическую
     * (так как результатом yql является статическая таблица с большим max_block_size).
     * См. [документацию](https://yt.yandex-team.ru/docs/description/dynamic_tables/dynamic_tables_mapreduce#konvertaciya-staticheskoj-tablicy-v-dinamicheskuyu)
     */
    private fun fixTableBlockSize(ytOperator: YtOperator, inputTable: YPath): YPath {
        val outputTable = YPath.simple(YtPathUtil.generateTemporaryPath())

        logger.info("Launching merge operation from $inputTable to $outputTable")
        ytOperator.yt.operations().mergeAndGetOp(
            MergeSpec.builder()
                // Необходимо сохранить сортированность
                .setMergeMode(MergeMode.SORTED)
                .setInputTables(inputTable)
                .setOutputTable(outputTable)
                .setJobIo(
                    JobIo(
                        TableWriterOptions(
                            DataSize.fromKiloBytes(256),
                            DataSize.fromMegaBytes(100)
                        )
                    )
                )
                .setAdditionalSpecParameters(mapOf(
                    // Для преобразования max_block_size
                    "force_transform" to YTree.booleanNode(true),
                    // Для конвертации в динамическую, у таблицы обязана остаться strict схема
                    "schema_inference_mode" to YTree.stringNode("from_input"),
                ))
                .build()
        ).awaitAndThrowIfNotSuccess(mergeAwaitTimeout)

        return outputTable
    }

    /**
     * @return пути к успешно скопированным таблицам
     */
    private fun transferTable(
        importCluster: YtCluster,
        importTablePath: YPath,
        outputPathByCluster: Map<YtCluster, YPath>,
    ): List<YtCluster> {
        // toList, чтобы зафиксировать порядок задач
        val outputClusters = outputPathByCluster.keys.toList()

        val transferConfigs = outputClusters.map { cluster ->
            val path = outputPathByCluster[cluster]
            TransferManagerJobConfig()
                .withInputCluster(importCluster.getName())
                .withInputTable(importTablePath.toString())
                .withOutputCluster(cluster.getName())
                .withOutputTable(path.toString())
        }

        logger.info("Querying transfer with tasks ${JsonUtils.toJson(transferConfigs)}")
        val transferTaskIds = transferManager.queryTransferManager(transferConfigs)
        val taskIdByCluster = outputClusters.zip(transferTaskIds).toMap()

        logger.info("Awaiting transfer tasks $transferTaskIds")
        val transferSuccess = transferManager.await(transferTaskIds, transferAwaitTimeout)

        return taskIdByCluster
            .mapNotNull { (cluster, taskId) ->
                if (transferSuccess[taskId] == true) {
                    cluster
                } else {
                    logger.error("Failed to transfer table to cluster ${cluster.getName()}")
                    null
                }
            }
    }

    private fun processOutput(cluster: YtCluster, tablePath: YPath) {
        val directory = tablePath.parent()
        val linkPath = getCurrentPath(directory.toString())

        val operator = ytProvider.getOperator(cluster)
        val table = YtTable(tablePath.toString())

        logger.info("Altering and mounting table $tablePath on cluster ${cluster.getName()}")
        operator.makeTableDynamic(table)
        operator.mount(tablePath, mountTimeout.toMillis().toInt())

        if (operator.yt.cypress().exists(YPath.simple(linkPath))) {
            // Существует старая таблица, заменяем линк
            val oldPath = getLinkTargetPath(operator, linkPath)
            logger.info("Changing link $linkPath from $oldPath to $tablePath on cluster ${cluster.getName()}")
            operator.yt.cypress().link(tablePath, YPath.simple(linkPath), true)

            logger.info("Unmounting old table $oldPath")
            operator.unmount(oldPath, mountTimeout.toMillis().toInt())
        } else {
            // Старой не существует, создаем линк
            logger.info("Creating link $linkPath to $tablePath on cluster ${cluster.getName()}")
            // нужен force, так как exists == false может быть у broken симлинка
            operator.yt.cypress().link(tablePath, YPath.simple(linkPath), true)
        }
    }

    private fun getImportCluster(): YtCluster? {
        for (cluster in config.importClusters) {
            if (isClusterAvailable(ytProvider, cluster)) {
                val operator = ytProvider.getOperator(cluster)
                if (!operator.yt.cypress().exists(YPath.simple(config.importTablePath))) {
                    logger.warn(
                        "Cluster ${cluster.getName()} is available, but table ${config.importTablePath} does not exist"
                    )
                    continue
                }

                val rowCount = operator.readTableRowCount(YtTable(config.importTablePath))
                if (rowCount == 0L) {
                    logger.warn(
                        "Cluster ${cluster.getName()} with table ${config.importTablePath} is available, " +
                            "but table is empty"
                    )
                    continue
                }

                return cluster
            }
        }

        return null
    }
}
