package ru.yandex.direct.jobs.slowlogs;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.SerializationFeature;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.AbstractBufferedLogbrokerWriter;
import ru.yandex.direct.cloud.mdb.mysql.api.ICloudMdbMySqlApi;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.ClusterRawInfo;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetClustersListRequest;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetClustersListResponse;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TestingOnly;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerProperties;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerPropertiesImpl;
import ru.yandex.direct.jobs.configuration.MySqlClustersParametersSource;
import ru.yandex.direct.jobs.slowlogs.readers.MySqlMdbClusterSlowLogsReader;
import ru.yandex.direct.jobs.slowlogs.readers.SlowLogRecord;
import ru.yandex.direct.jobs.slowlogs.readers.SlowLogRecordId;
import ru.yandex.direct.jobs.slowlogs.readers.SlowLogsRawRecordsResponse;
import ru.yandex.direct.jobs.slowlogs.transport.MySqlClusterInfo;
import ru.yandex.direct.jobs.slowlogs.transport.MySqlSlowLogsConverter;
import ru.yandex.direct.jobs.slowlogs.transport.MySqlSlowQueryLogRecord;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.mysql.slowlog.writer.states.MySqlClusterSlowLogsWriterFullStateInfo;
import ru.yandex.direct.mysql.slowlog.writer.states.MySqlClusterSlowLogsWriterStateInfo;
import ru.yandex.direct.mysql.slowlog.writer.states.MySqlClusterSlowLogsWriterStateProvider;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.sql.normalizer.QueryNormalizer;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.JsonUtilsWithMapper;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.kikimr.persqueue.producer.transport.message.inbound.ProducerWriteResponse;

import static ru.yandex.direct.jobs.configuration.MySqlSlowQueryLogsConfiguration.DEFAULT_CLOUD_MDB_MYSQL_API_BEAN_NAME;
import static ru.yandex.direct.jobs.configuration.MySqlSlowQueryLogsConfiguration.MYSQL_SLOW_QUERY_LOGS_WRITER_LOGBROKER_CLIENT_FACTORY_BEAN_NAME;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Джоба по перекладыванию slow query логов из кластеров MySQL в логброкер. Докуда переложила пишет ppc проперти
 * с префиксом <code>mysql_slow_query_logs_writer_</code>, для каждого кластера будет своя ppc проперти.
 * Чтобы запускать джобу локально через {@link ru.yandex.direct.jobs.DebugJobRunner}, нужно:
 * <ul>
 *     <li>Иметь в своем Яндекс-аккаунте доступ на чтение к внутреннему
 *     <a href="https://yc.yandex-team.ru/">Яндекc.Облаку</a> к каталогу direct-infra
 *     (<code>id: 'fooa07bcrr7souccreru'</code>). Если доступа нет, нужно его запросить у старших товарищей</li>
 *     <li>Получить <a
 *     href="https://oauth.yandex-team.ru/authorize?response_type=token&client_id=8cdb2f6a0dca48398c6880312ee2f78d">
 *     OAUTH-токен</a> для cloud api</li>
 *     <li>Записать его в файл <code>~/.direct-tokens/local_user_cloud_oauth_token</code></li>
 * </ul>
 */
@ParametersAreNonnullByDefault
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_VAMENSHOV,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2), needCheck = TestingOnly.class)
@Hourglass(periodInSeconds = 20, needSchedule = NonDevelopmentEnvironment.class)
@ParameterizedBy(parametersSource = MySqlClustersParametersSource.class)
public class MySqlSlowQueryLogsWriterJob extends DirectParameterizedJob<String> {
    private static final Logger logger = LoggerFactory.getLogger(MySqlSlowQueryLogsWriterJob.class);
    private static final int MAX_WORKING_SECONDS = 300;

    private final MySqlClustersParametersSource mySqlClustersParametersSource;
    private final ICloudMdbMySqlApi cloudMdbMySqlApi;
    private final LogbrokerClientFactoryFacade logbrokerClientCreatorFactory;
    private final MySqlSlowQueryLogsWriterLogbrokerPropertiesHolder logbrokerProperties;
    private final MySqlSlowQueryLogsWriterJobPropertiesHolder jobProperties;
    private final MySqlClusterSlowLogsWriterStateProvider clustersStatesProvider;
    private final Object locker = new Object();
    private boolean stopped = false;


    @Autowired
    public MySqlSlowQueryLogsWriterJob(
            MySqlClustersParametersSource mySqlClustersParametersSource,
            @Qualifier(DEFAULT_CLOUD_MDB_MYSQL_API_BEAN_NAME) ICloudMdbMySqlApi cloudMdbMySqlApi,
            @Qualifier(MYSQL_SLOW_QUERY_LOGS_WRITER_LOGBROKER_CLIENT_FACTORY_BEAN_NAME)
                    LogbrokerClientFactoryFacade logbrokerClientCreatorFactory,
            MySqlSlowQueryLogsWriterLogbrokerPropertiesHolder logbrokerProperties,
            MySqlSlowQueryLogsWriterJobPropertiesHolder jobProperties,
            MySqlClusterSlowLogsWriterStateProvider clustersStatesProvider) {
        this.mySqlClustersParametersSource = mySqlClustersParametersSource;
        this.cloudMdbMySqlApi = cloudMdbMySqlApi;
        this.logbrokerClientCreatorFactory = logbrokerClientCreatorFactory;
        this.logbrokerProperties = logbrokerProperties;
        this.jobProperties = jobProperties;
        this.clustersStatesProvider = clustersStatesProvider;
        JsonUtils.MAPPER.copy().disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
    }

    @Override
    public void execute() {
        stopped = false;
        MySqlSlowLogsLogbrokerWriter writer = null;
        LogbrokerProducerProperties producerProperties = null;
        try {
            String clusterName = mySqlClustersParametersSource.convertStringToParam(getParam());
            // Пробуем получить состояние записи слоу логов кластера, и проверяем, что они валидные и чтение логов
            // разрешено
            MySqlClusterSlowLogsWriterFullStateInfo clusterFullState = getAndCheckClusterFullState(clusterName);
            // Не получили состояние кластера, или с ним что-то не так - завершаем работу
            if (clusterFullState == null) {
                return;
            }
            // Идентификатор группы сообщений создаем как имя кластера плюс последняя цифра текущей секунды.
            // Так как идентификатор группы сообщений привязывается к партиции топика на две недели, а сама
            // партиция будет выбрана случайно (если параметр соединения group = 0), то дописывание номера
            // секунды к имени кластера при каждом запуске джобы, позволяет равномерно распределять нагрузку
            // с кластера на все партиции. Это, конечно, грозит нам потенциальным нарушением порядка записи
            // слоу логов, и даже потенциальной записью одного лога дважды, но порядок записи нам для аналитики
            // не важен, а с дубликатами можно успешно бороться в кликхаусе, куда в конечном итоге уедут логи,
            // с помощью ReplacingMergeTree
            // (https://clickhouse.com/docs/ru/engines/table-engines/mergetree-family/replacingmergetree/)
            // и с помощью запросов с модификатором словом FINAL
            // (https://clickhouse.com/docs/ru/sql-reference/statements/select/from/#select-from-final)
            String sourceId = clusterName + "_" + (LocalDateTime.now().getSecond() % 10);
            // Настраиваем поставщика писателей в логброкер. Группу указываем в 0, чтобы логброкер сам выбирал партицию
            // для записи. Повторы тоже ставим в 0, при ошибке джоба сама перезапустится через короткое время.
            producerProperties = LogbrokerProducerPropertiesImpl
                    .newBuilder()
                    .setRetries(0)
                    .setGroup(0)
                    .setHost(logbrokerProperties.getHost())
                    .setTimeoutSec(logbrokerProperties.getTimeout().toSeconds())
                    .setWriteTopic(logbrokerProperties.getTopic())
                    .setCompressionCodec(logbrokerProperties.getCompressionCodec())
                    .build();
            logger.info(String.format("Logbroker producer properties: %s", producerProperties));
            Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier =
                    logbrokerClientCreatorFactory.createProducerSupplier(producerProperties, sourceId);
            // Наш писатель в логброкер - наследник асинхронно-синхронного буферизированного писателя
            // AbstractBufferedLogbrokerWriter.
            writer = new MySqlSlowLogsLogbrokerWriter(
                    logbrokerProducerSupplier, logbrokerProperties.getTimeout(),
                    jobProperties.getRowsBufferSizeInBytes());
            // Нормализатор запроса. Умеет для mySQL запросов выдавать их нормальную форму, не зависящую от порядка
            // полей в селектах и апдейтах, и от значений параметров запроса.
            QueryNormalizer queryNormalizer = new QueryNormalizer(
                    200, 16 * 1024, 256 * 1024);

            MySqlClusterInfo cluster = new MySqlClusterInfo(
                    clusterFullState.getClusterState().getClusterId(), clusterName,
                    clusterFullState.getClusterState().getVersion());
            long nanos = System.nanoTime();
            SlowLogRecordId lastRecordId = new SlowLogRecordId(
                    clusterFullState.getClusterState().getTimestamp(),
                    clusterFullState.getClusterState().getPositionInTimestamp());
            int readRecordsCount = 0;
            boolean hasMoreRecords = true;
            // Крутимся, пока нас не попросили остановиться, или пока не прошло MAX_WORKING_SECONDS секунд, или пока
            // не закончатся данные для обработки
            while (!checkStopped() && hasMoreTime(nanos) && hasMoreRecords) {
                SlowLogsRawRecordsResponse cloudResponse = SlowLogsRawRecordsResponse.EMPTY;
                logger.debug("Trying to get slow query log records from cloud api");
                try {
                    // посылаем запрос за новой порцией слоу логов
                    cloudResponse = MySqlMdbClusterSlowLogsReader.getRecordsPortion(
                            cloudMdbMySqlApi, clusterFullState.getClusterState().getClusterId(),
                            lastRecordId, jobProperties.getCloudRowsInBatch());
                } catch (IOException ioex) {
                    // Если запрос упал, то выводим ошибку в лог, но работу не прекращаем, у нас, возможно, остались
                    // отправленные в логброкер, но неподтвержденные им сообщения. Жалко их терять и отправлять
                    // потом повторно, поэтому не пробрасываем ошибку дальше
                    logger.error(String.format("Something goes wrong, while receiving slow query logs from cluster %s",
                            clusterFullState.getClusterState().getClusterId()), ioex);
                }
                SlowLogRecord[] records = cloudResponse.getRecords();
                hasMoreRecords = cloudResponse.hasMoreRecords();
                logger.debug("Received {} slow query log records from cloud api, has more records: {}",
                        records.length, hasMoreRecords);
                logger.debug(String.format("Logbroker writer state before write:%n%s", writer.getStateAsString()));
                // Если новых данных не нашлось, можем переходить к завершению работы
                if (records.length == 0 || checkStopped()) {
                    break;
                }
                for (SlowLogRecord record : records) {
                    // Обогощаем запись для кликхауса сервисами, методами, requestId и нормальной формой запроса.
                    // Так же фильтруем запросы, в которых встречается вызов sleep(). Такие нам для аналитики
                    // точно не нужны
                    MySqlSlowQueryLogRecord recordToWrite =
                            MySqlSlowLogsConverter.convertRecord(
                                    cluster, record, queryNormalizer,
                                    jobProperties.getExcludeQueriesFilter(),
                                    jobProperties.isEnableSlowLogRecordRawTextField());
                    // Останавливаемся, если попросили или поток прервали
                    if (checkStopped()) {
                        break;
                    }
                    try {
                        // Пишем запись в логброкер. Если запись была отфильтрована выше, то recordToWrite будет null,
                        // но такую запись все равно отправляем в записыватель, вместе с идентификатором исходной
                        // записи. Реальная запись в логброкер при этом не пойдет, но идентификатор исходной записи
                        // не пропадет, и мы его сможем получить как обработанный у записывателя в логброкер позже
                        // (см. комментарии к методу write). Такой подход позволяет не городить огород с отдельным
                        // сохранением последних обработанных идентификаторов отфильтрованных строк. А идентификаторы
                        // отфильтрованных строк нужно сохранять обязательно, так как если мы получим от api набор
                        // логов, в котором все строки будут отфильтрованы, и не сохраним идентификатор последней,
                        // то мы попадем в вечный цикл. Потому что в следующий раз мы опять отправим запрос
                        // с предыдущего идентификатора, в ответ на который нам опять вернется набор логов, в котором
                        // все строки будут отфильтрованы. и т.д.
                        writer.write(recordToWrite, record.getRecordId());
                    } catch (ExecutionException | TimeoutException writeEx) {
                        // Если при записи произошла ошибка, то сохраняем в ppc проперти состояния кластера
                        // идентификатор последней подтвержденной записи и пробрасываем ошибку дальше
                        updateClusterState(writer, clusterName, clusterFullState);
                        throw writeEx;
                    }
                }
                logger.debug(String.format("Logbroker writer state after write:%n%s", writer.getStateAsString()));
                lastRecordId = records[records.length - 1].getRecordId();
                readRecordsCount += records.length;
                // Сохраняем в ppc проперти состояния кластера идентификатор последней подтвержденной записи
                clusterFullState = updateClusterState(writer, clusterName, clusterFullState);
            }
            // Все что успели и могли - прочитали и отправили на запись
            try {
                // Ожидаем, пока все запишется в логброкер
                writer.waitUntilAllWritten();
            } catch (ExecutionException | TimeoutException writeEx) {
                // Если при записи произошла ошибка, то сохраняем в ppc проперти состояния кластера
                // идентификатор последней подтвержденной записи и пробрасываем ошибку дальше
                updateClusterState(writer, clusterName, clusterFullState);
                throw writeEx;
            }
            logger.debug(String.format("Logbroker writer state before stop:%n%s", writer.getStateAsString()));
            // Сохраняем в ppc проперти состояния кластера идентификатор последней подтвержденной записи
            updateClusterState(writer, clusterName, clusterFullState);
            // Сравниваем идентификаторы последней прочитанной записи и последней сохраненной. Если ошибок не было,
            // то они должны совпасть
            SlowLogRecordId lastWrittenRecordId = writer.getLastWrittenMessageId();
            if (readRecordsCount > 0 && lastRecordId.compareTo(lastWrittenRecordId) != 0) {
                // А если не совпали, то пишем предупреждение в лог
                logger.warn(String.format(
                        "%d records read, but last read record id ('%s') not equals last write record id ('%s)",
                        readRecordsCount,
                        JsonUtilsWithMapper.JSON_UTILS_WITH_HUMAN_READABLE_DATES.toJson(lastRecordId),
                        JsonUtilsWithMapper.JSON_UTILS_WITH_HUMAN_READABLE_DATES.toJson(lastWrittenRecordId)));
            }
        } catch (InterruptedException intex) {
            Thread.currentThread().interrupt();
        } catch (Throwable ex) {
            logger.error("Something goes wrong while moving mysql slow query logs from cloud api to logbroker", ex);
        } finally {
            // Закрываем записыватель в логброкер
            if (writer != null) {
                // Как выяснилось, может кидать эксепшн, если инициализация была неудачной, поэтому оборачиваем
                // в try..catch (см https://st.yandex-team.ru/DIRECT-160212)
                try {
                    writer.close();
                } catch (Throwable t) {
                    logger.warn(String.format("Exception thrown while closing logbroker producer. " +
                            "Logbroker producer properties: %s", producerProperties), t);
                }
            }
            logger.info("Job for param {} finished", getParam());
        }
    }

    /**
     * Считывает идентификатор последней подтвержденной записи в логброкер и сохраняет его в ppc проперти
     * состояния кластера, если оно не изменилось с момента последнего чтения, а если изменилось, то кидает ошибку.
     * @param writer записыватель сообщений в логброкер
     * @param clusterName имя кластера
     * @param currentClusterFullState текущее состояние кластера из ppc проперти
     * @return новое состояние кластера из сохраненной ppc проперти (обновится, если только предыдущее состояние
     * не менялось с момента прочтения. Если оно менялось, то будет выброшено исключение). Это нужно, чтобы в случае
     * ручного изменения значения ppc проперти иметь возможность перезапускать джобу от нового состояния, а не
     * просто перетирать значения исправленные человеком.
     */
    private MySqlClusterSlowLogsWriterFullStateInfo updateClusterState(
            MySqlSlowLogsLogbrokerWriter writer, String clusterName,
            MySqlClusterSlowLogsWriterFullStateInfo currentClusterFullState) {
        SlowLogRecordId lastWrittenRecordId = writer.getLastWrittenMessageId();
        return writeState(clusterName, currentClusterFullState, lastWrittenRecordId);
    }

    private MySqlClusterSlowLogsWriterFullStateInfo getAndCheckClusterFullState(String clusterName)
            throws IOException, InterruptedException {
        MySqlClusterSlowLogsWriterFullStateInfo clusterFullState;
        try {
            clusterFullState = clustersStatesProvider.getClusterState(clusterName);
        } catch (Throwable t) {
            logger.error(String.format("Unable to get mysql slow logs writer cluster state property '%s', exit",
                    clustersStatesProvider.createPropertyName(clusterName)), t);
            return null;
        }
        if (clusterFullState == null) {
            logger.info("MySQL slow logs writer cluster state property '{}' is not exists or empty, trying to create",
                    clustersStatesProvider.createPropertyName(clusterName));

            ClusterRawInfo cluster = getClusterRawInfo(clusterName);

            clusterFullState = createNewClusterState(clusterName, cluster.getId());
            logger.info(
                    "MySQL slow logs writer cluster state property for cluster '{}' successfully created '{}': '{}'",
                    clusterName, clustersStatesProvider.createPropertyName(clusterName),
                    clusterFullState.getCompareAndSwapValue());
        }
        if (!clusterFullState.getClusterState().isAvailable()) {
            logger.warn("cluster '{}' is not available (ppc property '{}'='{}'), exit",
                    clusterName, clustersStatesProvider.createPropertyName(clusterName),
                    JsonUtilsWithMapper.JSON_UTILS_WITH_HUMAN_READABLE_DATES.toJson(clusterFullState));
            return null;
        }
        return clusterFullState;
    }

    /**
     * Получает информацию о кластере из cloud api по его имени. Если из апи кластеров не вернется, или у первого
     * вернувшегося имя не совпадет с заданным, то выбросит ошибку
     * @param clusterName имя кластера
     * @return Кластер по заданному имени
     * @throws IOException если что-то пошло не так при общении с cloud api
     * @throws InterruptedException если поток прервали во время ожидания результатов от api
     */
    @NotNull
    private ClusterRawInfo getClusterRawInfo(String clusterName) throws IOException, InterruptedException {
        logger.info("Trying to get mySQL cluster '{}' info from cloud api", clusterName);
        String filter = String.format("name='%s'", clusterName);
        GetClustersListRequest request = new GetClustersListRequest(
                jobProperties.getDirectInfraCloudFolderId(), 1, filter, null);
        GetClustersListResponse response = cloudMdbMySqlApi.getClustersList(request);
        ClusterRawInfo[] clusters = response.getElements();
        if (clusters == null || clusters.length == 0) {
            throw new IllegalStateException(String.format(
                    "Unable to get mySQL cluster '%s' info from cloud api. Empty cluster list received",
                    clusterName));
        }
        ClusterRawInfo cluster = clusters[0];
        if (!cluster.getName().equals(clusterName)) {
            throw new IllegalArgumentException(String.format(
                    "Cluster name '%s' from cloud api not equals to cluster name '%s' from job params",
                    cluster.getName(), clusterName));
        }
        return cluster;
    }

    /**
     * Создает начальное состояние кластера в ppc проперти.
     * @param clusterName имя кластера
     * @param clusterId идентификатор кластера
     * @return Начальное состояние кластера, если его создание прошло успешно. Если на момент сохранения начального
     * состояния в соответствующей ppc проперти уже что-то будет, то выбросится исключение
     */
    @NotNull
    private MySqlClusterSlowLogsWriterFullStateInfo createNewClusterState(
            String clusterName, String clusterId) {
        Instant from = LocalDateTime.now().toLocalDate().minusDays(jobProperties.getNewClusterLogsStartDaysAgo())
                .atStartOfDay(ZoneId.systemDefault()).toInstant();

        MySqlClusterSlowLogsWriterStateInfo newClusterState = new MySqlClusterSlowLogsWriterStateInfo(
                clusterId, 1, from, -1, true);

        MySqlClusterSlowLogsWriterFullStateInfo clusterFullState =
                new MySqlClusterSlowLogsWriterFullStateInfo(null, newClusterState);

        clusterFullState = clustersStatesProvider.compareAndSwap(clusterName, clusterFullState);

        if (clusterFullState == null) {
            throw new IllegalStateException(String.format(
                    "MySQL slow logs writer cluster state property '%s' has changed outside job, exit",
                    clustersStatesProvider.createPropertyName(clusterName)));
        }
        return clusterFullState;
    }

    /**
     * Сохраняет в ppc проперти состояния кластера идентификатор последней подтвержденной записи в логброкер.
     * Если в момент записи текущее состояние кластера не будет соответствовать currentState, то будет выброшена ошибка.
     * Такое поведение нужно, чтобы дать человеку возможность ручного перезапуска джобы с нового состояния
     * @param clusterName имя кластера
     * @param currentState последнее прочитанное состояние кластера
     * @param lastWrittenRecordId идентификатор последней подтвержденной записи в логброкер
     * @return новое состояние кластера из сохраненной ppc проперти, (обновится, если только на момент записи оно
     * соответствовало currentState, иначе будет выброшена ошибка)
     */
    private MySqlClusterSlowLogsWriterFullStateInfo writeState(
            String clusterName,
            MySqlClusterSlowLogsWriterFullStateInfo currentState,
            @Nullable SlowLogRecordId lastWrittenRecordId) {
        MySqlClusterSlowLogsWriterStateInfo oldClusterState = currentState.getClusterState();
        if (lastWrittenRecordId == null ||
                lastWrittenRecordId.compareTo(new SlowLogRecordId(oldClusterState.getTimestamp(),
                        oldClusterState.getPositionInTimestamp())) == 0) {
            return currentState;
        }
        MySqlClusterSlowLogsWriterStateInfo newClusterState = new MySqlClusterSlowLogsWriterStateInfo(
                oldClusterState.getClusterId(), oldClusterState.getVersion(),
                lastWrittenRecordId.getTimestamp(), lastWrittenRecordId.getPositionInTimestamp(), true);
        MySqlClusterSlowLogsWriterFullStateInfo newStateWithOldCasValue = new MySqlClusterSlowLogsWriterFullStateInfo(
                currentState.getCompareAndSwapValue(), newClusterState);
        logger.info("Trying to set new cluster '{}' offset: '{}'", clusterName, lastWrittenRecordId);
        MySqlClusterSlowLogsWriterFullStateInfo newFullState =
                clustersStatesProvider.compareAndSwap(clusterName, newStateWithOldCasValue);
        if (newFullState == null) {
            throw new IllegalStateException(String.format(
                    "MySQL slow logs writer cluster state property '%s' has changed outside job, exit",
                    clustersStatesProvider.createPropertyName(clusterName)));
        }
        logger.info("Offset: '{}' successfully set for cluster '{}'", lastWrittenRecordId, clusterName);
        return newFullState;
    }

    private boolean hasMoreTime(long nanos) {
        double seconds = (System.nanoTime() - nanos) / 1_000_000_000.0;
        return seconds < MAX_WORKING_SECONDS;
    }

    private boolean checkStopped() {
        synchronized (locker) {
           return stopped || Thread.currentThread().isInterrupted();
        }
    }

    @Override
    public void onShutdown() {
        synchronized (locker) {
            stopped = true;
        }
        super.onShutdown();
    }

    private static class MySqlSlowLogsLogbrokerWriter
            extends AbstractBufferedLogbrokerWriter<MySqlSlowQueryLogRecord, SlowLogRecordId> {
        public MySqlSlowLogsLogbrokerWriter(
                Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier,
                Duration logbrokerTimeout, long maxBytesOnTheFly) {
            super(logbrokerProducerSupplier, logbrokerTimeout, maxBytesOnTheFly);
        }

        @Override
        protected byte[] convertToBytes(MySqlSlowQueryLogRecord mySqlSlowQueryLogRecord) {
            String jsonRecord = JsonUtilsWithMapper.JSON_UTILS_WITH_HUMAN_READABLE_DATES
                    .toJson(mySqlSlowQueryLogRecord);
            return jsonRecord.getBytes(StandardCharsets.UTF_8);
        }

        @Override
        protected void acceptLogbrokerResponse(ProducerWriteResponse producerWriteResponse, SlowLogRecordId recordId) {
            if (producerWriteResponse.isAlreadyWritten()) {
                logger.error(String.format("Message with timestamp %s and position %d was already written",
                        recordId.getTimestamp(), recordId.getPositionInTimestamp()));
            }
        }
    }
}
