package ru.yandex.direct.jobs.slowlogs.readers;

import java.io.IOException;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.cloud.mdb.mysql.api.ICloudMdbMySqlApi;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetLogRecordsListRequest;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.GetLogRecordsListResponse;
import ru.yandex.direct.cloud.mdb.mysql.api.transport.LogTypeEnum;
import ru.yandex.direct.mysql.slowlog.parser.ParsedSlowLogRawRecord;
import ru.yandex.direct.mysql.slowlog.parser.SlowLogRawRecordParser;

public class MySqlMdbClusterSlowLogsReader {
    private static final Logger logger = LoggerFactory.getLogger(MySqlMdbClusterSlowLogsReader.class);

    /**
     * Получает очередную порцию записей slow query лога для кластера с идентификатором mdbClusterId от cloud-api.
     * Так как выяснилось, что внутри апи облака таймстемпы записей округляются до секунд, и поиск записи с таймстемпом
     * не кратным секунде может приводить к ошибкам (@see <a href="https://st.yandex-team.ru/MDBSUPPORT-4996">
     *     MDBSUPPORT-4996: [ MySQL ] Rest api (и gRPC) некорректно фильтрует слоу-логи по времени</a>),
     * то данная реализация при расчете идентификатора записи округляет её таймстемп до секунды, и считает её
     * позицию внутри этой секунды. То есть первая запись в секунде будет иметь позицию 0, следующая 1, и так далее.
     * Это позволяет посылать запросы в апи, начиная каждый запрос с начала секунды, пролистывая количество записей,
     * уже полученных внутри этой секунды, и таким образом обходить ошибку.
     * @param cloudApi реализация интерфейса api для работы с облаком
     * @param mdbClusterId идентификатор кластера, от которого нужно получить логи
     * @param lastRecordId идентификатор последней полученной записи
     * @param rowsLimit максимальное количество записей для получения
     * @return Очередная порция распарсеных записей slow query лога для кластера
     * с идентификатором mdbClusterId от cloud-api, начиная с позиции lastRecordId
     * @throws IOException если что-то пошло не так в io
     * @throws InterruptedException если поток прервали и пора закругляться
     */
    public static SlowLogsRawRecordsResponse getRecordsPortion(
            ICloudMdbMySqlApi cloudApi,
            String mdbClusterId, SlowLogRecordId lastRecordId, int rowsLimit)
            throws IOException, InterruptedException {
        String nextPageToken = null;
        if (lastRecordId.getPositionInTimestamp() >= 0) {
            nextPageToken = String.valueOf(lastRecordId.getPositionInTimestamp() + 1);
        }
        GetLogRecordsListRequest request = new GetLogRecordsListRequest(
                mdbClusterId, LogTypeEnum.SLOW_QUERY_LOG, lastRecordId.getTimestamp(), nextPageToken, rowsLimit);

        GetLogRecordsListResponse response = cloudApi.getLogRecordsList(request);

        SlowLogRecord[] records = new SlowLogRecord[response.getElements().length];

        Map<String, String> paramsMapBuffer = new HashMap<>(40);

        Instant lastTimestamp = lastRecordId.getTimestamp();
        int positionInTimestamp = lastRecordId.getPositionInTimestamp();
        int pos = 0;
        for (String rawRecord: response.getElements()) {
            String preparedRawRecord = rawRecord.replace("\\n", "\n")
                    .replace("\\t", "\t");
            ParsedSlowLogRawRecord parsedRecord;
            try {
                parsedRecord = SlowLogRawRecordParser.parseRawRecordText(preparedRawRecord, paramsMapBuffer);
            } catch (Throwable t) {
                logger.error(String.format("Unable to parse slow log record: %s", rawRecord), t);
                throw t;
            }
            Instant recordTimestamp = Instant.ofEpochSecond(parsedRecord.getTimestamp().getEpochSecond());

            if (recordTimestamp.equals(lastTimestamp)) {
                positionInTimestamp++;
            } else {
                positionInTimestamp = 0;
            }

            SlowLogRecordId recordId = new SlowLogRecordId(recordTimestamp, positionInTimestamp);
            records[pos++] = new SlowLogRecord(rawRecord, recordId, parsedRecord);
            lastTimestamp = recordTimestamp;
        }
        boolean hasMoreRecords = response.getNextPageToken() != null && !response.getNextPageToken().isEmpty();
        return new SlowLogsRawRecordsResponse(records, hasMoreRecords);
    }
}
