package ru.yandex.direct.jobs.slowlogs.transport;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.jobs.slowlogs.readers.SlowLogRecord;
import ru.yandex.direct.mysql.slowlog.parser.SlowLogRawRecordParser;
import ru.yandex.direct.sql.normalizer.QueryNormalizer;

public class MySqlSlowLogsConverter {
    private static final Logger logger = LoggerFactory.getLogger(MySqlSlowLogsConverter.class);

    public static MySqlSlowQueryLogRecord convertRecord(
            MySqlClusterInfo cluster, SlowLogRecord record, QueryNormalizer normalizer,
            String excludeQueriesFilter, boolean enableSlowLogRecordRawTextField) {
        try {
            String queryText = record.getParsedRecord().getQueryText();
            String lowerQueryText = queryText.toLowerCase();
            if (excludeQueriesFilter != null && !excludeQueriesFilter.isEmpty()
                    && lowerQueryText.contains(excludeQueriesFilter)) {
                return null;
            }
            MySqlQueryInfo queryInfo = extractQueryInfo(queryText, lowerQueryText, normalizer);
            return convertToFullRecord(cluster, queryInfo, record, enableSlowLogRecordRawTextField);
        } catch (Throwable t) {
            logger.error(String.format(
                    "Something goes wrong while converting query from cluster '%s' with id: '%s':%n%s",
                    cluster.getName(), cluster.getId(), record.getParsedRecord().getQueryText()), t);
        }
        return null;
    }

    private static MySqlSlowQueryLogRecord convertToFullRecord(
            MySqlClusterInfo cluster, MySqlQueryInfo queryInfo, SlowLogRecord record,
            boolean enableSlowLogRecordRawTextField) {
        String rawRecordText = enableSlowLogRecordRawTextField ? record.getRawText() : SlowLogRawRecordParser.EMPTY;
        return new MySqlSlowQueryLogRecord(
                cluster.getId(),
                cluster.getName(),
                cluster.getVersion(),
                record.getRecordId().getTimestamp().toEpochMilli(),
                record.getRecordId().getPositionInTimestamp(),
                queryInfo.getNormalForm(),
                queryInfo.getService(),
                queryInfo.getMethod(),
                queryInfo.getRequestId(),
                record.getParsedRecord().getTimestamp(),
                record.getParsedRecord().getQueryText(),
                record.getParsedRecord().getUserName(),
                record.getParsedRecord().getUserSecondName(),
                record.getParsedRecord().getUserHost(),
                record.getParsedRecord().getUserIp(),
                record.getParsedRecord().getConnectionId(),
                record.getParsedRecord().getSchema(),
                record.getParsedRecord().getLastErrorNumber(),
                record.getParsedRecord().getKilledCode(),
                record.getParsedRecord().getQueryTimeInSeconds(),
                record.getParsedRecord().getLockTimeInSeconds(),
                record.getParsedRecord().getRowsSentCount(),
                record.getParsedRecord().getRowsExaminedCount(),
                record.getParsedRecord().getRowsAffectedCount(),
                record.getParsedRecord().getBytesSentCount(),
                record.getParsedRecord().getTempTablesCount(),
                record.getParsedRecord().getTempTablesOnDiskCount(),
                record.getParsedRecord().getTempTablesSizesInBytes(),
                record.getParsedRecord().getTransactionId(),
                record.getParsedRecord().hasQcHit(),
                record.getParsedRecord().hasFullScan(),
                record.getParsedRecord().isHasFullJoin(),
                record.getParsedRecord().hasTempTables(),
                record.getParsedRecord().hasTempTablesOnDisk(),
                record.getParsedRecord().hasFileSort(),
                record.getParsedRecord().hasFileSortOnDisk(),
                record.getParsedRecord().getMergePassesCount(),
                record.getParsedRecord().getInnoDbIoReadOperationsCount(),
                record.getParsedRecord().getInnoDbIoReadBytesCount(),
                record.getParsedRecord().getInnoDbReadWaitInSeconds(),
                record.getParsedRecord().getInnoDbRecordsLockWaitInSeconds(),
                record.getParsedRecord().getInnoQbQueueWaitInSeconds(),
                record.getParsedRecord().getInnoDbPagesCountDistinct(),
                rawRecordText);
    }

    private static MySqlQueryInfo extractQueryInfo(
            String queryText, String lowerQueryText, QueryNormalizer normalizer) {
        String normalForm = normalizer.normalizeQuery(lowerQueryText);
        String[] serviceMethodAndRequestId = parseServiceMethodAndRequestId(queryText, lowerQueryText);
        long requestId = 0;
        if (serviceMethodAndRequestId[2] != null && !serviceMethodAndRequestId[2].isEmpty()) {
            try{
              requestId = Long.parseUnsignedLong(serviceMethodAndRequestId[2]);
            } catch (NumberFormatException numex) {
                logger.error(String.format(
                        "Unable to parse requestId from string '%s' from query:%n%s",
                        serviceMethodAndRequestId[2], queryText));

            }
        }
        return new MySqlQueryInfo(
                normalForm, serviceMethodAndRequestId[0], serviceMethodAndRequestId[1], requestId);
    }

    private static String[] parseServiceMethodAndRequestId(String queryText, String lowerQueryText) {
        // /* reqid:2522342514668344484:direct.script:ppcResendDomainsBS */
        String[] serviceMethodAndRequestId = new String[]{"", "", ""};
        String reqid = "/* reqid:";

        int index = lowerQueryText.indexOf(reqid);
        if (index < 0) {
            return serviceMethodAndRequestId;
        }
        int index2 = queryText.indexOf("*/", index + reqid.length());
        if (index2 < 0) {
            return serviceMethodAndRequestId;
        }
        String str = queryText.substring(index + reqid.length(), index2).trim();
        if (str.isBlank()) {
            return serviceMethodAndRequestId;
        }
        String[] parts = str.split(":");
        if (parts.length > 0) {
            serviceMethodAndRequestId[2] = parts[0];
        }

        if (parts.length < 2) {
            return serviceMethodAndRequestId;
        }

        if (parts.length > 2) {
            serviceMethodAndRequestId[0] = parts[1];
            serviceMethodAndRequestId[1] = parts[2];
        } else {
            serviceMethodAndRequestId[0] = parts[1];
        }

        return serviceMethodAndRequestId;
    }
}
