package ru.yandex.direct.jobs.statistics.activeorders;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.statistics.container.ClusterToFreshnessTimestamp;
import ru.yandex.direct.core.entity.statistics.model.ActiveOrderChanges;
import ru.yandex.direct.core.entity.statistics.model.YtHashBorders;
import ru.yandex.direct.core.entity.statistics.repository.OrderStatClusterChooseRepository;
import ru.yandex.direct.core.entity.statistics.repository.OrderStatRepository;
import ru.yandex.direct.dbschema.ppc.tables.CampOperationsQueue;
import ru.yandex.direct.dbschema.ppc.tables.Campaigns;
import ru.yandex.direct.dbschema.ppc.tables.WhenMoneyOnCampWas;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.grid.schema.yt.tables.CampaignstableDirect;
import ru.yandex.direct.grid.schema.yt.tables.OrderstatBs;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.common.db.PpcPropertyNames.ACTIVE_ORDERS_BATCHES_COUNT;
import static ru.yandex.direct.common.db.PpcPropertyNames.activeOrdersOrderStatBsCheventLogTimestamp;
import static ru.yandex.direct.core.entity.statistics.repository.OrderStatRepository.getYtHashBorders;
import static ru.yandex.direct.juggler.JugglerStatus.CRIT;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоб для обновления обновления показов, кликов, потраченных денег на кампании.
 * Импортирует данные о показах, кликах, потраченных деньгах кампнаний из динамических таблиц БК
 * 1) Из таблицы в БК {@link OrderstatBs} и таблицы {@link CampaignstableDirect} выбирает те кампании, в которых не
 * совпадают данные по показам, кликам и потраченным деньгам.
 * 2) Для на основе полученных считает дополнительные признаки кампаний: rollbacked, unarcived, newShows, finished,
 * moneyEnd
 * 3) Кампании с признаком unarcived отправляются в очередь на архивацию в таблицу {@link CampOperationsQueue}
 * 4) Для кампаний с признаком moneyEnd закрывается интервал в таблице {@link WhenMoneyOnCampWas}
 * 5) Для всех полученных капмнаний в mysql таблице директа обновляются данные о показах, кликах, потраченных
 * деньгах,
 * для кампаний с признаком newShows обновляется поле {@link Campaigns#LAST_SHOW_TIME},
 * для кампаний с признаком rollbacked обновляется поле {@link Campaigns#STATUS_BS_SYNCED},
 * для кампаний с признаком finished обновляется поле {@link Campaigns#STATUS_ACTIVE},
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 30),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_0, CheckTag.JOBS_RELEASE_REGRESSION})
@Hourglass(periodInSeconds = 300, needSchedule = NonDevelopmentEnvironment.class)
@ParameterizedBy(parametersSource = ActiveOrdersImportParametersSource.class)
public class ActiveOrdersImportJob extends DirectParameterizedJob<Integer> {

    private static final Logger logger = LoggerFactory.getLogger(ActiveOrdersImportJob.class);

    private final ActiveOrdersImportService activeOrdersImportService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final OrderStatRepository orderStatRepository;
    private final OrderStatClusterChooseRepository orderStatClusterChooseRepository;
    private final ShardHelper shardHelper;
    private final ActiveOrdersImportParametersSource parametersSource;
    private final int updateChunkSize;
    private Map<Integer, ActiveOrdersMetrics> activeOrdersMetrics;

    private static final long CRITICAL_BS_CHEVENT_LOG_DELAY = Duration.ofMinutes(90).getSeconds();
    private static final long DEFAULT_BATCHES_COUNT = 4L;

    ActiveOrdersImportJob(ActiveOrdersImportService activeOrdersImportService,
                          PpcPropertiesSupport ppcPropertiesSupport,
                          OrderStatRepository orderStatRepository,
                          OrderStatClusterChooseRepository orderStatClusterChooseRepository,
                          ShardHelper shardHelper,
                          ActiveOrdersImportParametersSource parametersSource,
                          @Value("${statistics.activeorders.batch_size}") int updateChunkSize)
    {
        this.activeOrdersImportService = activeOrdersImportService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.orderStatRepository = orderStatRepository;
        this.orderStatClusterChooseRepository = orderStatClusterChooseRepository;
        this.shardHelper = shardHelper;
        this.parametersSource = parametersSource;
        this.updateChunkSize = updateChunkSize;
    }

    @Override
    public void execute() {
        initMetrics();
        var lastBsCheventLogTimestamp = importActiveOrdersAndGetLastCheventLogTimestamp();
        calculateJugglerStatus(lastBsCheventLogTimestamp);
    }

    /**
     * @return bsCheventLogTimestamp, с которым происзошло последнее успешное обновление. Если обновления не
     * произошло(например, кластер недоступен), то вернется предыдущее значение, если удалось обновить - текущее
     */
    private long importActiveOrdersAndGetLastCheventLogTimestamp() {
        int workerNum = parametersSource.convertStringToParam(getParam());

        PpcProperty<Long> orderStatBsCheventLogTimestampProp =
                ppcPropertiesSupport.get(activeOrdersOrderStatBsCheventLogTimestamp(getParam()));

        var previousOrderStatBsCheventLogTimestamp = orderStatBsCheventLogTimestampProp.getOrDefault(0L);

        var clustersToFreshnessTimestampsShuffled =
                orderStatClusterChooseRepository.getClustersByPriority(previousOrderStatBsCheventLogTimestamp);

        if (clustersToFreshnessTimestampsShuffled.isEmpty()) {
            logger.warn("No suitable cluster");
            return previousOrderStatBsCheventLogTimestamp;
        }

        logger.info("Worker num: {}, cluster to freshness timestamp: {}", workerNum,
                clustersToFreshnessTimestampsShuffled);

        var allClusters = mapList(clustersToFreshnessTimestampsShuffled, ClusterToFreshnessTimestamp::getCluster);
        importActiveOrdersBatches(workerNum, allClusters);

        // Кластеры отсортированны по убыванию timestamp'а, поэтому макимальный timestamp будет у первого элемента
        var currentOrderStatBsCheventLogTimestamp = clustersToFreshnessTimestampsShuffled.get(0).getTimestamp();

        orderStatBsCheventLogTimestampProp.set(currentOrderStatBsCheventLogTimestamp);
        return currentOrderStatBsCheventLogTimestamp;
    }

    void importActiveOrdersBatches(int workerNum, List<YtCluster> clusters) {
        long batchesCount = ppcPropertiesSupport.get(ACTIVE_ORDERS_BATCHES_COUNT).getOrDefault(DEFAULT_BATCHES_COUNT);
        long ytHashMaxValue = orderStatRepository.getCampaignsYtHashMaxValue(clusters.get(0));
        long workersCount = parametersSource.getAllParamValues().size();

        for (int batchNum = 0; batchNum < batchesCount; ++batchNum) {
            YtHashBorders ytHashBorders = getYtHashBorders(
                    workerNum, workersCount, batchNum, batchesCount, ytHashMaxValue);
            logger.info("Starting iteration: worker num - {}, batch num - {}, YT hash borders - {}",
                    workerNum, batchNum, ytHashBorders);

            if (ytHashBorders == null) {
                logger.info("Empty batch, skipping");
                continue;
            }

            logger.info("Getting changes from YT");
            var activeOrderChanges = orderStatRepository.getChangedActiveOrders(ytHashBorders, clusters);

            logger.info("Received {} changes, start importing", activeOrderChanges.size());
            shardHelper.groupByShard(activeOrderChanges, ShardKey.CID, ActiveOrderChanges::getCid)
                    .forEach((shard, changes) -> {
                        // на тестовых средах одна и та же кампания без проблем может оказаться на разных шардах,
                        // поэтому если у изменения кампании шард из YT таблички не соответствует реальному из ppcdict,
                        // то отбрасываем его
                        List<ActiveOrderChanges> changesFiltered = changes.stream()
                                .filter(change -> shard == change.getShard())
                                .collect(Collectors.toList());

                        // делаем чанкование на верхнем уровне, так как не все методы обновления внутри себя его реализуют
                        for (var chunk : Iterables.partition(changesFiltered, updateChunkSize)) {
                            activeOrdersImportService.importActiveOrders(shard, chunk, activeOrdersMetrics.get(shard));
                        }
                    });

            logger.info("Import finished");
        }
    }

    private void initMetrics() {
        if (Objects.isNull(activeOrdersMetrics)) {
            this.activeOrdersMetrics = shardHelper.dbShards()
                    .stream()
                    .collect(Collectors.toMap(Function.identity(), ActiveOrdersMetrics::new));
        }
    }

    private void calculateJugglerStatus(long lastBsCheventLogTimestamp) {
        var delay = System.currentTimeMillis() / 1000 - lastBsCheventLogTimestamp;
        if (delay > CRITICAL_BS_CHEVENT_LOG_DELAY) {
            setJugglerStatus(CRIT, String.format("Very large lag of attribute bs_chevent_log of table OrderStat. " +
                    "Critical %d now %d", CRITICAL_BS_CHEVENT_LOG_DELAY, delay));
        }
    }
}
