package ru.yandex.direct.jobs.statistics.activeorders;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.common.log.container.LogActiveOrdersData;
import ru.yandex.direct.common.log.service.LogActiveOrdersService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.WhenMoneyOnCampWasRepository;
import ru.yandex.direct.core.entity.campoperationqueue.CampOperationQueueRepository;
import ru.yandex.direct.core.entity.campoperationqueue.model.CampQueueOperation;
import ru.yandex.direct.core.entity.campoperationqueue.model.CampQueueOperationName;
import ru.yandex.direct.core.entity.statistics.container.ProceededActiveOrder;
import ru.yandex.direct.core.entity.statistics.model.ActiveOrderChanges;
import ru.yandex.direct.core.entity.statistics.repository.ActiveOrdersRepository;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.tables.Campaigns;
import ru.yandex.direct.dbschema.ppc.tables.WhenMoneyOnCampWas;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.INTERNAL_DISTRIB;
import static ru.yandex.direct.currency.Money.MICRO_MULTIPLIER_SCALE;


/**
 * Сервис обрабатывает изменения на кампании:
 * 1) логирует изменения для экспорта их в clickhouse
 * 2) обновляет таблицу {@link WhenMoneyOnCampWas} для кампаний, на которых закончились деньги
 * 3) обновляет таблицу {@link CampQueueOperation} для кампаний, для которых проихозошел откат кликов
 * 4) обновляет таблицу {@link Campaigns}
 */

@Service
@ParametersAreNonnullByDefault
class ActiveOrdersImportService {
    private static final Logger logger = LoggerFactory.getLogger(ActiveOrdersImportService.class);

    private static final Long EPSILON = 1L;
    private static final int UPDATE_CHUNK_SIZE = 1000;
    private final CampaignRepository campaignRepository;
    private final WhenMoneyOnCampWasRepository whenMoneyOnCampWasRepository;
    private final CampOperationQueueRepository campOperationQueueRepository;
    private final WalletMoneyCalculator walletMoneyCalculator;
    private final ActiveOrdersRepository activeOrdersRepository;
    private final LogActiveOrdersService logActiveOrdersService;

    private final PpcProperty<Boolean> resendChangedCampaignsProp;

    ActiveOrdersImportService(
            CampaignRepository campaignRepository,
            PpcPropertiesSupport ppcPropertiesSupport,
            WhenMoneyOnCampWasRepository whenMoneyOnCampWasRepository,
            CampOperationQueueRepository campOperationQueueRepository,
            WalletMoneyCalculator walletMoneyCalculator,
            ActiveOrdersRepository activeOrdersRepository,
            LogActiveOrdersService logActiveOrdersService
    ) {
        this.campaignRepository = campaignRepository;
        this.resendChangedCampaignsProp =
                ppcPropertiesSupport.get(PpcPropertyNames.ACTIVE_ORDERS_RESEND_CHANGED_CAMPAIGNS);
        this.whenMoneyOnCampWasRepository = whenMoneyOnCampWasRepository;
        this.campOperationQueueRepository = campOperationQueueRepository;
        this.walletMoneyCalculator = walletMoneyCalculator;
        this.activeOrdersRepository = activeOrdersRepository;
        this.logActiveOrdersService = logActiveOrdersService;
    }

    /**
     * @param shard               номер шарда
     * @param activeOrdersChanges список изменений по кампаниям
     * @param activeOrdersMetrics метрики Solomon'а
     */
    void importActiveOrders(int shard, List<ActiveOrderChanges> activeOrdersChanges,
                            ActiveOrdersMetrics activeOrdersMetrics) {
        logActiveOrdersChanged(activeOrdersChanges);

        var moneyState = walletMoneyCalculator.initState(shard, activeOrdersChanges);

        var proceededActiveOrders = processOrders(shard, activeOrdersChanges, activeOrdersMetrics);
        updateOrders(shard, proceededActiveOrders, activeOrdersMetrics);

        resendCampaignsWithChangedMoneyState(shard, activeOrdersMetrics, moneyState, proceededActiveOrders);
    }

    private List<ProceededActiveOrder> processOrders(int shard,
                                                     List<ActiveOrderChanges> activeOrdersChanges,
                                                     ActiveOrdersMetrics activeOrdersMetrics) {
        return activeOrdersChanges.stream()
                .map(activeOrdersChange -> processOrder(activeOrdersChange, activeOrdersMetrics))
                .collect(toList());
    }

    ProceededActiveOrder processOrder(ActiveOrderChanges activeOrderChanges,
                                      ActiveOrdersMetrics activeOrdersMetrics) {
        var activeOrdersProceeded =
                new ProceededActiveOrder(
                        activeOrderChanges.getCid(),
                        activeOrderChanges.getType(),
                        activeOrderChanges.getNewShows(),
                        activeOrderChanges.getNewClicks(),
                        getNewSumSpent(activeOrderChanges),
                        activeOrderChanges.getUnits(),
                        activeOrderChanges.getNewSumSpentUnits());

        switch (activeOrderChanges.getType()) {
            case INTERNAL_DISTRIB:
                // Для дистрибуционных кампаний не считаем окончание
                calculateIfCampaignRollbackedOrUnarchived(activeOrderChanges, activeOrdersProceeded,
                        activeOrdersMetrics);
                break;
            case INTERNAL_FREE:
                // Расчёты для бесплатных внутренних кампаний основаны на юнитах (показы, клики, дни) вместо денег
                calculateIfInternalFreeCampaignRollbackedOrUnarchived(activeOrderChanges, activeOrdersProceeded,
                        activeOrdersMetrics);
                calculateIfInternalFreeCampaignFinished(activeOrderChanges, activeOrdersProceeded, activeOrdersMetrics);
                break;
            default:
                calculateIfCampaignRollbackedOrUnarchived(activeOrderChanges, activeOrdersProceeded,
                        activeOrdersMetrics);
                calculateIfCampaignFinished(activeOrderChanges, activeOrdersProceeded, activeOrdersMetrics);
        }
        calculateIfCampaignHasNewShows(activeOrderChanges, activeOrdersProceeded, activeOrdersMetrics);
        return activeOrdersProceeded;
    }

    private void calculateIfCampaignRollbackedOrUnarchived(ActiveOrderChanges activeOrderChanges,
                                                           ProceededActiveOrder proceededActiveOrder,
                                                           ActiveOrdersMetrics activeOrdersMetrics) {
        if (activeOrderChanges.getOldSumSpent() - activeOrderChanges.getNewSumSpent() > EPSILON) {
            processRollbackedOrder(activeOrderChanges, proceededActiveOrder, activeOrdersMetrics);
        }
    }

    private void calculateIfInternalFreeCampaignRollbackedOrUnarchived(ActiveOrderChanges activeOrderChanges,
                                                                       ProceededActiveOrder proceededActiveOrder,
                                                                       ActiveOrdersMetrics activeOrdersMetrics) {
        if (activeOrderChanges.getOldSumSpentUnits() - activeOrderChanges.getNewSumSpentUnits() > 0) {
            processRollbackedOrder(activeOrderChanges, proceededActiveOrder, activeOrdersMetrics);
        }
    }

    private void processRollbackedOrder(ActiveOrderChanges activeOrderChanges,
                                        ProceededActiveOrder proceededActiveOrder,
                                        ActiveOrdersMetrics activeOrdersMetrics) {
        proceededActiveOrder.setRollbacked(true);
        activeOrdersMetrics.incrementRollbackedCamps();
        if (CampaignsArchived.Yes.getLiteral().equals(activeOrderChanges.getArchived())) {
            proceededActiveOrder.setUnarchived(true);
            activeOrdersMetrics.incrementUnarchCamps();
        }
    }

    private void calculateIfCampaignFinished(ActiveOrderChanges activeOrderChanges,
                                             ProceededActiveOrder proceededActiveOrder,
                                             ActiveOrdersMetrics activeOrdersMetrics) {
        if (activeOrderChanges.getSum() > activeOrderChanges.getOldSumSpent()
                && activeOrderChanges.getSum() <= activeOrderChanges.getNewSumSpent()) {
            proceededActiveOrder.setFinished(true);
            activeOrdersMetrics.incrementFinishedCamps();
            if (activeOrderChanges.getWalletCid() == 0L) {
                proceededActiveOrder.setMoneyEnd(true);
                activeOrdersMetrics.incrementMoneyEndCamps();
            }
        }
    }

    private void calculateIfInternalFreeCampaignFinished(ActiveOrderChanges activeOrderChanges,
                                                         ProceededActiveOrder proceededActiveOrder,
                                                         ActiveOrdersMetrics activeOrdersMetrics) {
        if (activeOrderChanges.getUnits() > activeOrderChanges.getOldSumSpentUnits()
                && activeOrderChanges.getUnits() <= activeOrderChanges.getNewSumSpentUnits()) {
            proceededActiveOrder.setFinished(true);
            activeOrdersMetrics.incrementFinishedCamps();
        }
    }

    private void calculateIfCampaignHasNewShows(ActiveOrderChanges activeOrderChanges,
                                                ProceededActiveOrder proceededActiveOrder,
                                                ActiveOrdersMetrics activeOrdersMetrics) {
        if (activeOrderChanges.getNewShows() > activeOrderChanges.getOldShows()) {
            proceededActiveOrder.setNewShows(true);
            activeOrdersMetrics.incrementNewShowsCamps();
        }
    }


    private void updateWhenMoneyOnCampaignWas(int shard, List<ProceededActiveOrder> proceededActiveOrder) {
        List<Long> cidsWithMoneyEnd = proceededActiveOrder.stream()
                .filter(ProceededActiveOrder::isMoneyEnd)
                .map(ProceededActiveOrder::getCid)
                .collect(toList());

        if (cidsWithMoneyEnd.isEmpty()) {
            return;
        }
        logger.info("Shard {}: {} campaigns with money end", shard, cidsWithMoneyEnd.size());
        whenMoneyOnCampWasRepository.closeInterval(shard, cidsWithMoneyEnd);
    }

    private void unarcCampaigns(int shard, List<ProceededActiveOrder> activeOrdersProceeded) {
        List<CampQueueOperation> unarcCampaigns = activeOrdersProceeded.stream()
                .filter(ProceededActiveOrder::isUnarchived)
                .map(activeOrderProceeded ->
                        new CampQueueOperation()
                                .withCid(activeOrderProceeded.getCid())
                                .withCampQueueOperationName(CampQueueOperationName.UNARC))
                .collect(toList());

        if (unarcCampaigns.isEmpty()) {
            return;
        }
        logger.info("Shard {}: {} campaigns unarchived", shard, unarcCampaigns.size());
        campOperationQueueRepository.addCampaignQueueOperations(shard, unarcCampaigns);
    }

    private void logActiveOrdersChanged(List<ActiveOrderChanges> activeOrdersChanges) {
        var logActiveOrdersDataList = activeOrdersChanges.stream()
                .map(activeOrderChanges ->
                        new LogActiveOrdersData(activeOrderChanges.getOrderId())
                                .withShows(activeOrderChanges.getNewShows())
                                .withClicks(activeOrderChanges.getNewClicks())
                                .withSpentUnits(activeOrderChanges.getNewSumSpentUnits())
                                .withCost(activeOrderChanges.getCost())
                                .withCostCur(activeOrderChanges.getCostCur())
                                .withUpdateTime(LocalDateTime.ofInstant(Instant.ofEpochSecond(activeOrderChanges.getUpdateTime()), ZoneId.of("UTC"))))
                .collect(toList());
        logActiveOrdersService.logActiveOrders(logActiveOrdersDataList);
    }

    void updateOrders(int shard, List<ProceededActiveOrder> proceededActiveOrders,
                      ActiveOrdersMetrics activeOrdersMetrics) {
        updateWhenMoneyOnCampaignWas(shard, proceededActiveOrders);
        unarcCampaigns(shard, proceededActiveOrders);

        int updatedCampaigns = activeOrdersRepository.updateCampaigns(shard, proceededActiveOrders);
        activeOrdersMetrics.addUpdatedCampaigns(updatedCampaigns);
    }

    /**
     * Для внутренних дистрибуционных кампаний никогда не пишем значение меньше, чем выставленная кампании сумма
     * <p>
     * БК умеет начислять внутренним дистрибуционным кампаниям "условные" деньги так, что разница
     * {@code sum - sum_spent} никогда не бывает отрицательной. В директе такой логики нет, поэтому для таких
     * кампаний никогда не записываем в БД значение больше, чем {@code sum}, то есть если {@code sum_spent}
     * становится больше, чем {@code sum}, для записи берём значение {@code sum}.
     */
    private BigDecimal getNewSumSpent(ActiveOrderChanges activeOrderChanges) {
        return convertFromMicro(activeOrderChanges.getType() == INTERNAL_DISTRIB ?
                Math.min(activeOrderChanges.getSum(), activeOrderChanges.getNewSumSpent()) :
                activeOrderChanges.getNewSumSpent());
    }

    /**
     * Переотправить в БК кампании, у которых закончились или появились деньги
     */
    private void resendCampaignsWithChangedMoneyState(
            int shard,
            ActiveOrdersMetrics activeOrdersMetrics,
            WalletMoneyProcessState moneyState,
            List<ProceededActiveOrder> proceededActiveOrders
    ) {
        // получить кошельки, у которых изменилось наличие денег
        var changedWalletIds = walletMoneyCalculator.getChangedWalletIds(moneyState);
        var changedCampaignIds = campaignRepository.getStartedCampaignIdsByWalletIds(shard, changedWalletIds);
        var changedSimpleCampaigns = proceededActiveOrders.stream()
                .filter(ProceededActiveOrder::isMoneyEnd)
                .map(ProceededActiveOrder::getCid)
                .collect(toList());

        var allChangedIds = StreamEx.of(changedCampaignIds)
                .append(changedWalletIds)
                .append(changedSimpleCampaigns)
                .distinct().sorted().toList();
        activeOrdersMetrics.addResentCampaigns(allChangedIds.size());

        if (!allChangedIds.isEmpty() && resendChangedCampaignsProp.getOrDefault(false)) {
            for (var chunk : Iterables.partition(allChangedIds, UPDATE_CHUNK_SIZE)) {
                campaignRepository.updateStatusBsSynced(shard, chunk, StatusBsSynced.NO);
            }
        }
    }

    private BigDecimal convertFromMicro(long valInMicro) {
        return BigDecimal.valueOf(valInMicro, MICRO_MULTIPLIER_SCALE);
    }
}
