package ru.yandex.direct.jobs.statistics.activeorders

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.AutoOverdraftUtils
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.client.model.ClientAutoOverdraftInfo
import ru.yandex.direct.core.entity.client.repository.ClientRepository
import ru.yandex.direct.core.entity.statistics.model.ActiveOrderChanges
import ru.yandex.direct.currency.Currencies.EPSILON
import ru.yandex.direct.dbutil.model.ClientId
import java.math.BigDecimal
import java.math.BigDecimal.ZERO

data class WalletMoneyProcessState(
    val shard: Int,
    val wallets: Collection<WalletCampaign>,
    val clientsAutoOverdraftInfo: Map<Long, ClientAutoOverdraftInfo>,
    val moneyWas: Map<Long, Boolean>
)

@Component
open class WalletMoneyCalculator constructor(
    private val campaignRepository: CampaignRepository,
    private val clientRepository: ClientRepository,
) {
    companion object {
        private val logger = LoggerFactory.getLogger(WalletMoneyCalculator::class.java)
    }

    fun initState(shard: Int, activeOrdersChanges: List<ActiveOrderChanges>): WalletMoneyProcessState {
        val walletIds = activeOrdersChanges.map { it.walletCid }.filter { it > 0 }.distinct()
        val wallets = campaignRepository.getWalletsByWalletCampaignIds(shard, walletIds)

        val clientIds = wallets.map { ClientId.fromLong(it.clientId) }
        val clientsAutoOverdraftInfo = clientRepository.getClientsAutoOverdraftInfo(shard, clientIds)
            .associateBy { it.clientId }

        val moneyWas = calcHasMoney(shard, wallets, clientsAutoOverdraftInfo)

        return WalletMoneyProcessState(shard, wallets, clientsAutoOverdraftInfo, moneyWas)
    }

    fun getChangedWalletIds(state: WalletMoneyProcessState): List<Long> {
        val moneyHas = calcHasMoney(state.shard, state.wallets, state.clientsAutoOverdraftInfo)

        logWalletMoney(state.wallets, state.moneyWas, moneyHas)

        return state.wallets.filter { state.moneyWas[it.id] != moneyHas[it.id] }.map { it.id }
    }
    
    private fun logWalletMoney(
        wallets: Collection<WalletCampaign>,
        moneyWas: Map<Long, Boolean>,
        moneyHas: Map<Long, Boolean>
    ) {
        if (!logger.isInfoEnabled) return

        fun Boolean?.fmt() =
            when (this) {
                null -> "N"
                true -> "T"
                false -> "F"
            }
        for (chunk in wallets.chunked(50)) {
            val formatted = chunk.map { it.id }.map { "$it:${moneyWas[it].fmt()}${moneyHas[it].fmt()}" }
                .joinToString(", ")
            logger.info("wallet money change: $formatted")
        }
    }

    private fun calcHasMoney(
        shard: Int,
        wallets: Collection<WalletCampaign>,
        clientsAutoOverdraftInfo: Map<Long, ClientAutoOverdraftInfo>
    ): Map<Long, Boolean> {
        val walletsDebt = campaignRepository.getWalletsDebt(shard, wallets.map { it.id })
        val moneyHas = wallets.associateBy(
            { it.id },
            { calcHasMoneyForWallet(it, clientsAutoOverdraftInfo, walletsDebt[it.id]) }
        )
        return moneyHas
    }

    private fun calcHasMoneyForWallet(
        wallet: WalletCampaign,
        clientsAutoOverdraftInfo: Map<Long, ClientAutoOverdraftInfo>,
        walletDebt: BigDecimal?
    ): Boolean {
        val overdraftAddition = clientsAutoOverdraftInfo[wallet.clientId]?.let {
            AutoOverdraftUtils.calculateAutoOverdraftAddition(
                wallet.currency, wallet.sum, walletDebt?.negate() ?: ZERO, it
            )
        } ?: ZERO

        return wallet.sum - (walletDebt ?: ZERO) + overdraftAddition > EPSILON
    }
}
