package ru.yandex.direct.jobs.statistics.advq;

import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.direct.core.entity.statistics.AdvqHits;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static com.google.common.base.Preconditions.checkState;
import static java.lang.Long.parseLong;
import static ru.yandex.direct.jobs.statistics.advq.ExportAdvqHitsDeviceType.ALL;
import static ru.yandex.direct.jobs.statistics.advq.ExportAdvqHitsDeviceType.DESKTOP;
import static ru.yandex.direct.jobs.statistics.advq.ExportAdvqHitsDeviceType.PHONE;
import static ru.yandex.direct.jobs.statistics.advq.ExportAdvqHitsDeviceType.TABLET;

@Component
public class ExportAdvqDataConverter {

    private final GeoBaseHelper geoBaseHelper;

    @Autowired
    public ExportAdvqDataConverter(GeoBaseHelper geoBaseHelper) {
        this.geoBaseHelper = geoBaseHelper;
    }

    public List<AdvqHits> convert(Tuple2List<String, YTreeNode> advqHitsRaw) {
        Map<ExportAdvqHitsDeviceType, Map<Long, Long>> showsByTargetings = StreamEx.of(advqHitsRaw)
                .mapToEntry(Tuple2::get1, Tuple2::get2)
                .mapValues(YTreeNode::stringValue)
                .mapValues(this::parseShowsByRegionId)
                .mapKeys(ExportAdvqHitsDeviceType::fromAdvq)
                .mapValues(this::addShowsInChildrenToShows)
                .toMap();

        normalizeDeviceTypes(showsByTargetings);

        return convertToAdvqHits(showsByTargetings);
    }

    private void normalizeDeviceTypes(Map<ExportAdvqHitsDeviceType, Map<Long, Long>> showsByRegionIdByDeviceType) {
        checkState(!showsByRegionIdByDeviceType.containsKey(DESKTOP));

        Set<Long> allRegionIds = EntryStream.of(showsByRegionIdByDeviceType)
                .values()
                .flatMap(showsByRegionId -> showsByRegionId.keySet().stream())
                .toSet();

        Map<Long, Long> desktopShowsByRegionId = StreamEx.of(allRegionIds)
                .mapToEntry(regionId -> {
                    Long phoneShows = showsByRegionIdByDeviceType.get(PHONE).getOrDefault(regionId, 0L);
                    Long tabletShows = showsByRegionIdByDeviceType.get(TABLET).getOrDefault(regionId, 0L);
                    Long allShows = showsByRegionIdByDeviceType.get(ALL).getOrDefault(regionId, 0L);

                    checkState(allShows >= phoneShows + tabletShows);

                    return allShows - phoneShows - tabletShows;
                })
                .toMap();

        showsByRegionIdByDeviceType.put(DESKTOP, desktopShowsByRegionId);
        showsByRegionIdByDeviceType.remove(ALL);
    }

    /**
     * К числу показов каждого региона из {@code showsByRegionId} прибавляет показы в дочерных регионах.
     * <p>
     * Например, было: { 225 => 1, 1 => 1, 213 => 1 }, станет { 225 => 3, 1 => 2, 213 => 1 }
     */
    private Map<Long, Long> addShowsInChildrenToShows(Map<Long, Long> showsByRegionId) {
        PriorityQueue<Long> queue = new PriorityQueue<>(Comparator.reverseOrder());
        queue.addAll(showsByRegionId.keySet());
        Set<Long> processedRegionIds = new HashSet<>();

        while (!queue.isEmpty()) {
            Long regionId = queue.poll();
            if (processedRegionIds.contains(regionId)) {
                continue;
            }

            List<Integer> parentRegionIds = geoBaseHelper.getParentRegionIds(regionId);
            Long toAdd = 0L;
            for (Integer parentRegionId : parentRegionIds) {
                Long regionIdToProcess = Long.valueOf(parentRegionId);

                Long newShows = showsByRegionId.getOrDefault(regionIdToProcess, 0L) + toAdd;
                showsByRegionId.put(regionIdToProcess, newShows);

                if (!processedRegionIds.contains(regionIdToProcess)) {
                    toAdd = newShows;
                    processedRegionIds.add(regionIdToProcess);
                }
            }
        }

        return showsByRegionId;
    }

    private Map<Long, Long> parseShowsByRegionId(String showsByRegionIdByDeviceTypeRaw) {
        return StreamEx.of(showsByRegionIdByDeviceTypeRaw.split(" "))
                .map(showsByRegion -> showsByRegion.split(":"))
                .mapToEntry(
                        regionIdAndShows -> parseLong(regionIdAndShows[0]),
                        regionIdAndShows -> parseLong(regionIdAndShows[1]))
                .toMap();
    }

    private List<AdvqHits> convertToAdvqHits(Map<ExportAdvqHitsDeviceType, Map<Long, Long>> showsByTargetings) {
        return EntryStream.of(showsByTargetings)
                .flatMapValues(showsByRegionId -> showsByRegionId.entrySet().stream())
                .map(entry -> new AdvqHits()
                        .withDeviceType(entry.getKey().convertToCore())
                        .withRegionId(entry.getValue().getKey())
                        .withShows(entry.getValue().getValue()))
                .toList();
    }
}
