package ru.yandex.direct.jobs.statistics.advq;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.statistics.AdvqHits;
import ru.yandex.direct.core.entity.statistics.repository.AdvqHitsRepository;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static ru.yandex.direct.common.db.PpcPropertyNames.EXPORT_ADVQ_HITS_LAST_PROCESSED_WEEK_NUMBER;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.ytwrapper.YtUtils.isClusterAvailable;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 3),
        notifications = @OnChangeNotification(
                recipient = {NotificationRecipient.CHAT_API_CLEAN_MONITORING},
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        ),
        tags = {DIRECT_PRIORITY_2})
@Hourglass(cronExpression = "0 0 * * * ?")
public class ExportAdvqHitsJob extends DirectJob {

    private static final YtCluster ADVQ_TABLE_CLUSTER = YtCluster.HAHN;

    private static final Logger logger = LoggerFactory.getLogger(ExportAdvqHitsJob.class);

    private final YtProvider ytProvider;

    private final PpcProperty<String> lastProcessedWeekNumberProperty;
    private final AdvqHitsRepository advqHitsRepository;
    private final ExportAdvqDataConverter exportAdvqDataConverter;

    @Autowired
    public ExportAdvqHitsJob(YtProvider ytProvider,
                             PpcPropertiesSupport ppcPropertiesSupport,
                             AdvqHitsRepository advqHitsRepository,
                             ExportAdvqDataConverter exportAdvqDataConverter) {
        this.ytProvider = ytProvider;
        this.advqHitsRepository = advqHitsRepository;
        this.exportAdvqDataConverter = exportAdvqDataConverter;

        lastProcessedWeekNumberProperty = ppcPropertiesSupport.get(EXPORT_ADVQ_HITS_LAST_PROCESSED_WEEK_NUMBER);
    }

    @Override
    public void execute() {
        String lastProcessedWeekNumberRaw = lastProcessedWeekNumberProperty.get();
        if (lastProcessedWeekNumberRaw == null) {
            logger.info("No work to do");
            return;
        }

        if (!isClusterAvailable(ytProvider, ADVQ_TABLE_CLUSTER)) {
            logger.warn("Cluster not available");
            return;
        }

        AdvqHitsTablePath tablePath = AdvqHitsTablePath.fromPropertyValue(lastProcessedWeekNumberRaw);

        boolean success = doWorkIfCan(tablePath.nextWeek()) || doWorkIfCan(tablePath.nextYear());
        if (success) {
            logger.info("AdvqHits updated");
        }
    }

    private boolean doWorkIfCan(AdvqHitsTablePath tablePath) {
        YPath tableYPath = tablePath.toYPath();

        if (!tableExists(tableYPath)) {
            return false;
        }

        exportDataFrom(tableYPath);
        updateProperty(tablePath);

        return true;
    }

    private boolean tableExists(YPath tableYPath) {
        return ytProvider.get(ADVQ_TABLE_CLUSTER).cypress().exists(tableYPath);
    }

    private void exportDataFrom(YPath tableYPath) {
        List<AdvqHits> dataToExport = exportAdvqDataConverter.convert(getAdvqHitsRaw(tableYPath));
        advqHitsRepository.deleteAllAdvqHits();
        advqHitsRepository.addAdvqHits(dataToExport);
    }

    private Tuple2List<String, YTreeNode> getAdvqHitsRaw(YPath tableYPath) {
        return Cf.wrap(ytProvider.get(ADVQ_TABLE_CLUSTER).cypress().get(tableYPath).asMap()).entries();
    }

    private void updateProperty(AdvqHitsTablePath tablePath) {
        lastProcessedWeekNumberProperty.set(tablePath.getPropertyValue());
    }
}
