package ru.yandex.direct.jobs.statistics.auctionstat;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.common.db.PpcPropertyNames.AUCTIONSTAT_PREPARED_CHEVENT_TIME;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Hourglass(periodInSeconds = 3 * 3600, needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 7),
        tags = {DIRECT_PRIORITY_1, CheckTag.YT, CheckTag.GROUP_INTERNAL_SYSTEMS},
        needCheck = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class AuctionStatPrepareJob extends DirectJob {
    private static final String QUERY_RESOURCE = "classpath:///statistics/auctionstat/prepare.yql";

    private final YtProvider ytProvider;
    private final AuctionStatService auctionStatService;

    private final String directDbRoot;
    private final YtTable bsStatTable;
    private final YtTable preparedDataTable;
    private final List<YtCluster> clusters;

    private final PpcProperty<String> syncTimeProp;

    @Autowired
    public AuctionStatPrepareJob(
            YtProvider ytProvider,
            AuctionStatService auctionStatService,
            PpcPropertiesSupport ppcPropertiesSupport,
            DirectConfig directConfig
    )
    {
        this.ytProvider = ytProvider;
        this.auctionStatService = auctionStatService;

        var cfg = directConfig.getBranch("statistics.auctionstat");
        this.directDbRoot = cfg.getString("direct_db_root");
        this.bsStatTable = new YtTable(cfg.getString("bs_stat_table"));
        this.preparedDataTable = new YtTable(cfg.getString("prepared_data_table"));
        this.clusters = mapList(cfg.getStringList("clusters"), YtCluster::parse);

        this.syncTimeProp = ppcPropertiesSupport.get(AUCTIONSTAT_PREPARED_CHEVENT_TIME);
    }

    @Override
    public void execute() {
        var yql = LiveResourceFactory.get(QUERY_RESOURCE).getContent();
        var clusterData = auctionStatService.getSuitableCluster(clusters, bsStatTable, syncTimeProp.get());

        if (clusterData == null) {
            throw new IllegalStateException("Suitable cluster not found");
        }

        var operator = ytProvider.getOperator(clusterData.cluster(), YtSQLSyntaxVersion.SQLv1);
        var tmpResult = new YtTable(preparedDataTable.getPath() + ".tmp");
        operator.yqlExecute(
                yql,
                directDbRoot, bsStatTable.getPath(), tmpResult.getPath()
        );
        operator.writeTableStringAttribute(tmpResult, AuctionStatService.SYNC_ATTRIBUTE, clusterData.syncAttr());
        operator.getYt().cypress()
                .move(tmpResult.ypath(), preparedDataTable.ypath(), true);
        syncTimeProp.set(clusterData.syncAttr());
    }
}
