package ru.yandex.direct.jobs.statistics.auctionstat;

import java.time.Duration;
import java.time.Instant;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtTable;

@Service
public class AuctionStatService {
    static final String SYNC_ATTRIBUTE = "last_sync_time_bs-chevent-log";
    private static final Logger logger = LoggerFactory.getLogger(AuctionStatService.class);
    private static final Duration MAX_STAT_DELAY = Duration.ofHours(7);

    private final YtProvider ytProvider;

    public AuctionStatService(YtProvider ytProvider) {
        this.ytProvider = ytProvider;
    }

    /**
     * На каждом кластере из clusters таблцы table читает атрибут last_sync_time_bs-chevent-log
     * Выбирает кластер с максимальным значением атрибута. Если это значение меньше чем previousSyncTime, то
     * возвращается null
     * Если кластер не удалось выбрать - вернется нет
     */
    @Nullable
    ClusterData getSuitableCluster(List<YtCluster> clusters, YtTable table, String previousSyncTime) {
        ClusterData result = clusters.stream()
                .map(cluster -> tryGetClusterData(cluster, table))
                .flatMap(Optional::stream)
                .max(Comparator.comparing(ClusterData::syncAttr))
                .orElseThrow(() -> new IllegalStateException("Suitable cluster not found"));

        if (previousSyncTime != null && result.syncAttr.compareTo(previousSyncTime) <= 0) {
            logger.info("Previous table was fresher");
            return null;
        } else {
            var delay = System.currentTimeMillis() - Instant.parse(result.syncAttr).toEpochMilli();
            if (delay > MAX_STAT_DELAY.toMillis()) {
                throw new IllegalStateException("Freshest cluster " + result.cluster +
                        " is too outdated: " + result.syncAttr);
            }
            return result;
        }
    }

    Optional<ClusterData> tryGetClusterData(YtCluster cluster, YtTable table) {
        try {
            var ytOperator = ytProvider.getOperator(cluster);
            if (!ytOperator.exists(table)) {
                logger.warn("Stat table not exists on cluster {}", cluster);
                return Optional.empty();
            }
            var attr = ytOperator.tryReadTableStringAttribute(table, SYNC_ATTRIBUTE);
            if (attr.isEmpty()) {
                logger.warn("Can't read attribute on cluster {}", cluster);
            } else {
                return Optional.of(new ClusterData(cluster, attr.get()));
            }
        } catch (RuntimeException e) {
            logger.warn("Error on cluster " + cluster, e);
        }
        return Optional.empty();
    }

    static class ClusterData {
        private YtCluster cluster;
        private String syncAttr;

        ClusterData(YtCluster cluster, String syncAttr) {
            this.cluster = cluster;
            this.syncAttr = syncAttr;
        }

        YtCluster cluster() {
            return cluster;
        }

        String syncAttr() {
            return syncAttr;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ClusterData that = (ClusterData) o;
            return cluster == that.cluster &&
                    Objects.equals(syncAttr, that.syncAttr);
        }

        @Override
        public int hashCode() {
            return Objects.hash(cluster, syncAttr);
        }

        @Override
        public String toString() {
            return "ClusterData{" +
                    "cluster=" + cluster +
                    ", syncAttr='" + syncAttr + '\'' +
                    '}';
        }
    }
}
