package ru.yandex.direct.jobs.statistics.auctionstat;


import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.statistics.container.ProcessedAuctionStat;
import ru.yandex.direct.core.entity.statistics.repository.BsAuctionStatRepository;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.cypress.RangeLimit;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;

import static java.time.format.DateTimeFormatter.ofPattern;
import static ru.yandex.direct.common.db.PpcPropertyNames.auctionstatUpdateCheventLogTimestamp;
import static ru.yandex.direct.common.db.PpcPropertyNames.bsAuctionStatTime;
import static ru.yandex.direct.juggler.JugglerStatus.CRIT;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.YT;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба читает подготовленную джобой {@link AuctionStatPrepareJob} таблицу с данными для обновления bs_auction_stat,
 * создает аналогичную таблицу с постфиксом _getShard(), в которую записывает все данные для нужного шарда.
 * Созданную тублицу читает по чанкам, для каждого чанка делает обновление в mysql
 */
@Hourglass(periodInSeconds = 3600, needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3, minutes = 5),
        tags = {DIRECT_PRIORITY_1, YT, GROUP_INTERNAL_SYSTEMS},
        needCheck = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class AuctionStatUpdateJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(AuctionStatUpdateJob.class);

    private final YtProvider ytProvider;
    private final BsAuctionStatRepository bsAuctionStatRepository;
    private final List<YtCluster> clusters;
    private final String preparedDataTableName;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final AuctionStatMetrics auctionStatMetrics;
    private final AuctionStatService auctionStatService;

    private static final int CHUNK_SIZE = 50000;
    private static final long CRITICAL_BS_CHEVENT_LOG_DELAY = Duration.ofHours(7).getSeconds();
    private static final DateTimeFormatter PPC_PROP_DATETIME_FORMATTER = ofPattern("yyyyMMddHHmmss");
    private static final RelaxedWorker RELAXED_WORKER = new RelaxedWorker(3.0);

    AuctionStatUpdateJob(
            YtProvider ytProvider,
            BsAuctionStatRepository bsAuctionStatRepository,
            PpcPropertiesSupport ppcPropertiesSupport,
            AuctionStatMetrics auctionStatMetrics,
            DirectConfig directConfig, AuctionStatService auctionStatService)
    {
        this.ytProvider = ytProvider;
        this.bsAuctionStatRepository = bsAuctionStatRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.auctionStatMetrics = auctionStatMetrics;
        this.clusters = mapList(directConfig.getStringList("statistics.auctionstat.clusters"), YtCluster::parse);
        this.preparedDataTableName = directConfig.getString("statistics.auctionstat.prepared_data_table");
        this.auctionStatService = auctionStatService;
    }

    @Override
    public void execute() {
        int shard = getShard();
        var lastBsCheventLogTimestamp = importAuctionStatAndGetLastCheventLogTimestamp(shard);
        calculateJugglerStatus(lastBsCheventLogTimestamp);
    }

    private String importAuctionStatAndGetLastCheventLogTimestamp(int shard) {
        var auctionStatUpdateCheventLogTimestampProp =
                ppcPropertiesSupport.get(auctionstatUpdateCheventLogTimestamp(shard));
        var previousSuccessCheventLogTimestamp = auctionStatUpdateCheventLogTimestampProp.getOrDefault("0");

        var clusterData = auctionStatService.getSuitableCluster(clusters, new YtTable(preparedDataTableName),
                previousSuccessCheventLogTimestamp);

        if (Objects.isNull(clusterData)) {
            logger.info("No suitable cluster");
            return previousSuccessCheventLogTimestamp;
        }

        logger.info("Shard {}, cluster data {}", shard, clusterData);
        YtOperator operator = ytProvider.getOperator(clusterData.cluster());

        String shardedPreparedTableName = preparedDataTableName + "_" + shard;
        mergePreparedTableToShardedPreparedTable(shard, operator, shardedPreparedTableName);

        YtTable shardedPreparedYtTable = new YtTable(shardedPreparedTableName);
        var statTimeInstant = Instant.now();
        var statTime = LocalDateTime.ofInstant(statTimeInstant, MSK);

        try (TraceProfile ignore = Trace.current().profile("read_sharded_table_and_write_in_mysql")) {
            operator.readTableSnapshot(shardedPreparedYtTable, new AuctionStatTableRow(),
                    row -> convertRowToContainer(row, statTime), this::updateBsAuctionStat, CHUNK_SIZE);
        }

        ppcPropertiesSupport.get(bsAuctionStatTime(shard)).set(statTime.format(PPC_PROP_DATETIME_FORMATTER));
        auctionStatUpdateCheventLogTimestampProp.set(clusterData.syncAttr());
        return clusterData.syncAttr();
    }

    private ProcessedAuctionStat convertRowToContainer(AuctionStatTableRow auctionStatTableRow,
            LocalDateTime statTime)
    {
        return new ProcessedAuctionStat.Builder()
                .withPid(auctionStatTableRow.getPid())
                .withPhraseId(auctionStatTableRow.getPhraseId())
                .withClicks(auctionStatTableRow.getClicks())
                .withPclicks(auctionStatTableRow.getPclicks())
                .withShows(auctionStatTableRow.getShows())
                .withPshows(auctionStatTableRow.getPshows())
                .withRank(auctionStatTableRow.getRank())
                .withStatTime(statTime)
                .build();
    }

    /**
     * Выбирает из preparedTable все строки, у которых ключ шард равен getShard() и записывает в таблицу с именем
     * имя_изначальной_таблицы + _ + номер_шарда
     */
    private void mergePreparedTableToShardedPreparedTable(int shard, YtOperator operator,
            String shardedPreparedTableName)
    {
        try (TraceProfile ignored = Trace.current().profile("merging_to_sharded_table")) {
            YPath preparedTablePath = getPreparedTableYpath(shard);

            YPath shardedPreparedTableYpath = YPath.simple(shardedPreparedTableName);
            logger.info("Shard {}, start merging table", shard);
            operator.getYt().operations().mergeAndGetOp(Cf.list(preparedTablePath), shardedPreparedTableYpath).await();
        }
    }

    private YPath getPreparedTableYpath(int shard) {
        RangeLimit rangeLimit1 = new RangeLimit(Cf.list(YTree.integerNode(shard)), -1, -1);
        RangeLimit rangeLimit2 = new RangeLimit(Cf.list(YTree.integerNode(shard + 1)), -1, -1);
        return YPath.simple(preparedDataTableName).withRange(rangeLimit1, rangeLimit2);
    }

    private void updateBsAuctionStat(List<ProcessedAuctionStat> list) {
        int shard = getShard();
        RELAXED_WORKER.runAndRelax(() -> bsAuctionStatRepository.updateBsAuctionStat(shard, list));
        auctionStatMetrics.addUpdatedRows(shard, list.size());
        logger.info("Shard {} updated {} rows", shard, list.size());

    }

    private void calculateJugglerStatus(String lastBsCheventLogTimestamp) {
        var delay = System.currentTimeMillis() / 1000 - Instant.parse(lastBsCheventLogTimestamp).getEpochSecond();
        if (delay > CRITICAL_BS_CHEVENT_LOG_DELAY) {
            setJugglerStatus(CRIT,
                    String.format("Very large lag of attribute bs_chevent_log of table %s. Critical %d now %d",
                            preparedDataTableName, CRITICAL_BS_CHEVENT_LOG_DELAY, delay));
        }
    }
}
