package ru.yandex.direct.jobs.statistics.rollbacknotifications;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.validator.routines.EmailValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.util.LocaleGuard;
import ru.yandex.direct.core.entity.campaign.model.Wallet;
import ru.yandex.direct.core.entity.campaign.service.MailTextCreatorService;
import ru.yandex.direct.core.entity.campaign.service.WalletService;
import ru.yandex.direct.core.entity.changes.model.StatRollbackDropTypeRow;
import ru.yandex.direct.core.entity.changes.repository.StatRollbacksRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.sender.YandexSenderClient;
import ru.yandex.direct.sender.YandexSenderException;
import ru.yandex.direct.sender.YandexSenderTemplateParams;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба получает данные по изменениям статистики от БК, начиная с момента последнего получения этих изменений,
 * и рассылает соответствующие уведомления клиентам
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 1),
        notifications = @OnChangeNotification(method = NotificationMethod.TELEGRAM,
                recipient = NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        ),
        needCheck = ProductionOnly.class,
        tags = {GROUP_INTERNAL_SYSTEMS}
)
@Hourglass(periodInSeconds = 600, needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class StatRollbackEmailSenderJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(StatRollbackEmailSenderJob.class);

    private static final DateTimeFormatter OUTPUT_FORMAT = DateTimeFormatter.ofPattern("dd.MM.yyyy");
    private static final DateTimeFormatter INPUT_FORMAT = DateTimeFormatter.ofPattern("yyyyMMdd");

    private static final EmailValidator emailValidator = EmailValidator.getInstance();

    private static final int ROW_LIMIT = 10_000;
    private static final String EMAIL_FOR_TESTING = "stat-rollbacks-test-letters@yandex-team.ru";
    private static final String LOGIN = "login";
    private static final String CLIENT_ID = "ClientID";
    private static final String SUM_WITH_CURRENCY = "sum";
    private static final String PERIOD = "period";
    private static final String CAMPAIGN_ID = "num";
    private static final String CAMPAIGNS_COUNT = "nums";
    private static final String CAMPAIGNS_COUNT_X1 = "nums1";

    private final StatRollbacksRepository statRollbacksRepository;
    private final YandexSenderClient senderClient;
    private final StatRollbacksMailTemplateResolver statRollbacksMailTemplateResolver;
    private final UserService userService;
    private final ClientService clientService;
    private final MailTextCreatorService mailTextCreatorService;
    private final WalletService walletService;

    @Autowired
    public StatRollbackEmailSenderJob(StatRollbacksRepository statRollbacksRepository,
                                      YandexSenderClient senderClient,
                                      StatRollbacksMailTemplateResolver statRollbacksMailTemplateResolver,
                                      MailTextCreatorService mailTextCreatorService,
                                      UserService userService, ClientService clientService,
                                      WalletService walletService) {
        this.statRollbacksRepository = statRollbacksRepository;
        this.senderClient = senderClient;
        this.statRollbacksMailTemplateResolver = statRollbacksMailTemplateResolver;
        this.userService = userService;
        this.clientService = clientService;
        this.mailTextCreatorService = mailTextCreatorService;
        this.walletService = walletService;
    }

    private static void logEvent(StatRollbackMailEvent event) {
        logger.info(JsonUtils.toJson(event));
    }

    private static String segment(LocalDate from, LocalDate to) {
        if (from.equals(to)) {
            return OUTPUT_FORMAT.format(from);
        }
        return OUTPUT_FORMAT.format(from) + "-" + OUTPUT_FORMAT.format(to);
    }

    static String getPeriod(List<String> periods) {
        List<LocalDate> dates = StreamEx.of(periods)
                .map(period -> LocalDate.parse(period, INPUT_FORMAT))
                .sorted().distinct()
                .toList();
        List<String> segments = new ArrayList<>();
        if (dates.size() < 3) {
            segments = mapList(dates, date -> segment(date, date));
        } else {
            LocalDate from = null;
            LocalDate to = null;
            for (LocalDate date : dates) {
                if (from == null) {
                    from = date;
                } else if (ChronoUnit.DAYS.between(to, date) > 1) {
                    segments.add(segment(from, to));
                    from = date;
                }
                to = date;
            }
            segments.add(segment(from, to));
        }

        return StreamEx.of(segments).joining(", ");
    }

    @Override
    public void execute() {
        Map<Long, Map<Long, StatRollbackDropTypeRow>> usersToStatRollbacks =
                statRollbacksRepository.getStatRollbacksData(getShard(), ROW_LIMIT);
        Map<Long, User> uidToUser = listToMap(userService.massGetUser(usersToStatRollbacks.keySet()), User::getId);
        List<ClientId> clientIds = mapList(uidToUser.values(), User::getClientId)
                .stream().distinct().collect(Collectors.toList());
        Map<ClientId, Currency> clientIdToCurrency = clientService.massGetWorkCurrency(clientIds);
        // оставляем только включенные кошельки в валюте клиента
        List<Wallet> wallets = walletService.massGetWallets(clientIds).stream()
                .filter(e -> e.getCampaignsCurrency().equals(clientIdToCurrency.get(e.getClientId())) && e.getEnabled())
                .collect(Collectors.toList());
        Map<ClientId, Boolean> clientIdToWalletEnable = new HashMap<>();
        // у клиента могут быть несколько активных кошельков, считаем что у таких клиентов кошелек включен
        wallets.forEach((wallet -> {
            clientIdToWalletEnable.putIfAbsent(wallet.getClientId(), wallet.getEnabled());
        }));

        usersToStatRollbacks.forEach((uid, rows) -> {
            User user = uidToUser.get(uid);
            rows.forEach((id, row) -> {
                Currency currency = clientIdToCurrency.get(user.getClientId());
                StatRollbacksMailNotificationType notificationType =
                        getNotificationType(row.getDropType(),
                                clientIdToWalletEnable.getOrDefault(user.getClientId(), false));
                sendMail(getShard(), user, id, currency.getCode(), row, notificationType);
            });
        });
    }

    private void sendMail(int shard, User user, Long id, CurrencyCode currencyCode,
                          StatRollbackDropTypeRow statRollbackDropTypeRow,
                          StatRollbacksMailNotificationType notificationType) {
        // при запуске джобы в тестовом окружении шлем письма на рассылку
        String email = Environment.getCached().isProduction() ? user.getEmail() : EMAIL_FOR_TESTING;
        if (!emailValidator.isValid(email)) {
            logEvent(StatRollbackMailEvent.statRollbackBadEmail(user.getClientId(), user, email));
            return;
        }
        String campSlug = statRollbacksMailTemplateResolver.resolveTemplateId(user.getLang(), notificationType);
        if (StringUtils.isEmpty(campSlug)) {
            logEvent(StatRollbackMailEvent.statRollbackEmptySlug(user.getClientId(), user, email));
            return;
        }
        Map<String, String> params = getParams(user, statRollbackDropTypeRow, currencyCode);
        YandexSenderTemplateParams templateParams = new YandexSenderTemplateParams.Builder()
                .withCampaignSlug(campSlug)
                .withToEmail(email)
                .withAsync(Boolean.TRUE)
                .withArgs(params)
                .build();
        try {
            if (senderClient.sendTemplate(templateParams, YandexSenderClient::isInvalidToEmail)) {
                logEvent(StatRollbackMailEvent.mailSent(notificationType, user.getClientId(), user, campSlug));
            } else {
                logEvent(StatRollbackMailEvent.statRollbackBadEmail(user.getClientId(), user, email));
            }
            statRollbacksRepository.deleteStatRollbacksDataRow(shard, id);
        } catch (YandexSenderException e) {
            logEvent(StatRollbackMailEvent.senderError(notificationType,
                    user.getClientId(),
                    user, campSlug, e.getMessage()));
        }
    }

    private String getSumWithCurrency(BigDecimal sum, CurrencyCode currencyCode, Language language) {
        if (currencyCode.equals(CurrencyCode.RUB)) {
            try (LocaleGuard ignored = LocaleGuard.fromLanguage(language)) {
                return StringUtils.removeEnd(mailTextCreatorService.formatMoney(sum, currencyCode), ".");
            }
        }
        Money money = Money.valueOf(sum, currencyCode);
        return mailTextCreatorService.getMoneyWithoutCurrency(money) + " " + currencyCode;
    }

    private StatRollbacksMailNotificationType getNotificationType(String dropType, Boolean isWalletEnable) {
        switch (dropType) {
            case "fraud":
                return (isWalletEnable ? StatRollbacksMailNotificationType.FRAUD_WITH_WALLET
                        : StatRollbacksMailNotificationType.FRAUD_WITHOUT_WALLET);
            case "chargeback":
                return (isWalletEnable ? StatRollbacksMailNotificationType.CHARGEBACK_WITH_WALLET
                        : StatRollbacksMailNotificationType.CHARGEBACK_WITHOUT_WALLET);
            case "error":
                return (isWalletEnable ? StatRollbacksMailNotificationType.ERROR_WITH_WALLET
                        : StatRollbacksMailNotificationType.ERROR_WITHOUT_WALLET);
            case "perror":
                return (isWalletEnable ? StatRollbacksMailNotificationType.PERROR_WITH_WALLET
                        : StatRollbacksMailNotificationType.PERROR_WITHOUT_WALLET);
            case "custom":
                return (isWalletEnable ? StatRollbacksMailNotificationType.CUSTOM_WITH_WALLET
                        : StatRollbacksMailNotificationType.CUSTOM_WITHOUT_WALLET);
            default:
                String error = "Unknown dropType: " + dropType;
                throw new IllegalArgumentException(error);
        }
    }

    private Map<String, String> getParams(User user, StatRollbackDropTypeRow statRollbackDropTypeRow,
                                          CurrencyCode currencyCode) {
        Map<String, String> params = new HashMap<>();
        params.put(LOGIN, user.getLogin());
        params.put(CLIENT_ID, user.getClientId().toString());
        params.put(SUM_WITH_CURRENCY, getSumWithCurrency(statRollbackDropTypeRow.getSumCur(), currencyCode,
                user.getLang()));
        params.put(PERIOD, getPeriod(statRollbackDropTypeRow.getPeriods()));
        params.put(CAMPAIGNS_COUNT_X1, ""); // jinja Рассылятора считает неприсланные переменные правдой
        params.put(CAMPAIGNS_COUNT, "");
        params.put(CAMPAIGN_ID, "");

        var campaignCount = statRollbackDropTypeRow.getCids().stream().distinct().count();
        if (campaignCount == 1) {
            params.put(CAMPAIGN_ID, statRollbackDropTypeRow.getCids().get(0));
        } else if (campaignCount > 20 && campaignCount % 10 == 1) {
            // для согласованности падежей в русском шаблоне: 21 (31 ... 991) вашей кампании
            params.put(CAMPAIGNS_COUNT_X1, String.valueOf(campaignCount));
        } else {
            params.put(CAMPAIGNS_COUNT, String.valueOf(campaignCount));
        }
        return params;
    }
}
