package ru.yandex.direct.jobs.suggestconversionprice;

import java.time.LocalDate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.suggestconversionprice.config.ConversionPriceJobConfig;
import ru.yandex.direct.jobs.suggestconversionprice.service.SuggestConversionPriceService;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static java.time.LocalDate.EPOCH;
import static java.time.LocalDate.now;
import static ru.yandex.direct.common.db.PpcPropertyNames.LAST_SUCCESSFUL_SUGGEST_CONVERSION_PRICE_UPDATE_DATE;
import static ru.yandex.direct.common.db.PpcPropertyNames.POPULATE_SUGGEST_CONVERSION_PRICE_DYN_TABLE_JOB_ENABLED;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM;

/**
 * Перекладывает данные из статической YT таблицы
 * {@link //home/ads/cpa_autobudget/scripts/suggest_conversion_price}
 * автобюджета в динамическую на стороне директа.
 */
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 4),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, DIRECT_SPB_SERVER_SIDE_TEAM},
        notifications = @OnChangeNotification(
                recipient = {NotificationRecipient.LOGIN_NINAZHEVTYAK, NotificationRecipient.LOGIN_SSDMITRIEV},
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(cronExpression = "0 0 * * * ?", needSchedule = NonDevelopmentEnvironment.class)
public class PopulateSuggestConversionPriceDynTableJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(PopulateSuggestConversionPriceDynTableJob.class);

    private final LocalDate todayDate = now();

    private final SuggestConversionPriceService suggestConversionPriceService;

    private final YtProvider ytProvider;
    private final PpcProperty<Boolean> jobEnabledProperty;
    private final PpcProperty<LocalDate> lastSuccessfulUpdateDate;

    private YtCluster sourceCluster;
    private YtCluster targetCluster;
    private String sourceTablePath;
    private String targetTablePath;

    @Autowired
    public PopulateSuggestConversionPriceDynTableJob(
            DirectConfig directConfig,
            YtProvider ytProvider,
            PpcPropertiesSupport ppcPropertiesSupport,
            SuggestConversionPriceService suggestConversionPriceService
    ) {
        this.ytProvider = ytProvider;
        this.suggestConversionPriceService = suggestConversionPriceService;

        setTablesInfoFromConfig(directConfig);

        jobEnabledProperty = ppcPropertiesSupport.get(POPULATE_SUGGEST_CONVERSION_PRICE_DYN_TABLE_JOB_ENABLED);
        lastSuccessfulUpdateDate = ppcPropertiesSupport.get(LAST_SUCCESSFUL_SUGGEST_CONVERSION_PRICE_UPDATE_DATE);
    }

    private void setTablesInfoFromConfig(DirectConfig directConfig) {
        ConversionPriceJobConfig jobConfig = new ConversionPriceJobConfig(directConfig);
        sourceCluster = YtCluster.parse(jobConfig.getSourceCluster());
        targetCluster = YtCluster.parse(jobConfig.getTargetCluster());
        sourceTablePath = getSourceTablePath(jobConfig.getSourceTableDirPath());
        targetTablePath = getTargetTablePath(ytProvider, jobConfig.getTargetTableRelativePath());
    }

    @Override
    public void execute() {
        if (!jobEnabledProperty.getOrDefault(false)) {
            logger.info("Job is not enabled");
            return;
        }

        if (lastSuccessfulUpdateDate.getOrDefault(EPOCH).isEqual(todayDate)) {
            logger.info("Table has already been updated today");
            return;
        }

        suggestConversionPriceService.importTable(sourceCluster, sourceTablePath,
                targetCluster, targetTablePath, lastSuccessfulUpdateDate, todayDate);
    }

    private String getTargetTablePath(YtProvider ytProvider,
                                      String targetTableFolder) {
        return YtPathUtil.generatePath(
                ytProvider.getClusterConfig(targetCluster).getHome(), relativePart(), targetTableFolder
        );
    }

    private String getSourceTablePath(String sourceTableDirPath) {
        String tableName = todayDate.minusDays(7) + "_" + todayDate.minusDays(1);
        return sourceTableDirPath + "/" + tableName + "/" + tableName;
    }

}
