package ru.yandex.direct.jobs.suggestconversionprice.repository;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.model.YtTableRow;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtUtils.isClusterAvailable;
import static ru.yandex.inside.yt.kosher.tables.YTableEntryTypes.YSON;

@Repository
public class SuggestConversionPriceRepository {

    private static final Logger logger = LoggerFactory.getLogger(SuggestConversionPriceRepository.class);
    private final YtProvider ytProvider;

    @Autowired
    public SuggestConversionPriceRepository(YtProvider ytProvider) {
        this.ytProvider = ytProvider;
    }

    public void importChunk(YtCluster sourceCluster, YtTable fromTable,
                            YtCluster targetCluster, YPath toTableYpath,
                            long chunkBegin, long chunkEnd) {
        YtOperator sourceYtOperator = ytProvider.getOperator(sourceCluster);

        try {
            List<YTreeMapNode> fromTableRows = readTableByRowRange(sourceYtOperator, fromTable, chunkBegin, chunkEnd);

            List<YTreeMapNode> rowsToWrite = mapList(fromTableRows, SuggestConversionPriceRepository::convertRow);
            insertRows(rowsToWrite, toTableYpath, targetCluster, ytProvider);
        } catch (Exception e) {
            logger.error("Got error while importing chunk from table {}[{}, {}] to table {}:",
                    fromTable.getName(), chunkBegin, chunkEnd, toTableYpath.name(), e);
            throw new IllegalStateException("Error while importing table " + fromTable.getPath() + " to table " + toTableYpath, e);
        }
    }

    public boolean isTableUnavailable(YPath tableYPath, YtCluster cluster) {
        Cypress cypress = ytProvider.get(cluster).cypress();
        if (!isClusterAvailable(ytProvider, cluster) || !cypress.exists(tableYPath)) {
            logger.error("Unable to get table {} from cluster {}", tableYPath, cluster.getName());
            return true;
        }

        return false;
    }

    public long getTableSize(YtCluster cluster, YtTable table) {
        YtOperator sourceYtOperator = ytProvider.getOperator(cluster);
        return sourceYtOperator.readTableRowCount(table);
    }

    private static List<YTreeMapNode> readTableByRowRange(YtOperator sourceYtOperator, YtTable fromTable,
                                                          long chunkBegin, long chunkEnd) {
        List<YTreeMapNode> rows = new ArrayList<>();
        sourceYtOperator.readTableByRowRange(
                fromTable, ytTableRow -> rows.add(ytTableRow.getData()), new YtTableRow(),
                chunkBegin, chunkEnd
        );
        return rows;
    }

    private static void insertRows(List<YTreeMapNode> rowsToWrite, YPath toTableYpath, YtCluster targetCluster,
                                   YtProvider ytProvider) {
        ytProvider.get(targetCluster).tables()
                .insertRows(toTableYpath, true, false, false, YSON, Cf.wrap(rowsToWrite).iterator());
    }

    private static YTreeMapNode convertRow(YTreeMapNode node) {
        YTreeBuilder builder = YTree.mapBuilder();
        builder.key("AttributionType").value(node.getInt("AttributionType"));
        builder.key("CategoryID").value(node.getLong("CategoryID"));
        builder.key("CategoryName").value(node.getString("CategoryName"));
        builder.key("ClicksCount").value(node.getDouble("ClicksCount"));
        builder.key("GoalType").value(node.getString("GoalType"));
        builder.key("MeanPrice").value(node.getDouble("MeanPrice"));
        builder.key("RegionID").value(node.getInt("RegionID"));
        builder.key("RegionName").value(node.getString("RegionName"));
        builder.key("SignificantCategoryID").value(node.getLong("SignificantCategoryID"));
        builder.key("StdMeanPrice").value(node.getDouble("StdMeanPrice"));
        return builder.buildMap();
    }
}
