package ru.yandex.direct.jobs.suggestconversionprice.service;

import java.time.LocalDate;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.jobs.suggestconversionprice.repository.SuggestConversionPriceRepository;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static java.lang.Math.min;

@Service
@ParametersAreNonnullByDefault
public class SuggestConversionPriceService {
    private static final Logger logger = LoggerFactory.getLogger(SuggestConversionPriceService.class);
    private static final int CHUNK_SIZE = 50_000;

    private final SuggestConversionPriceRepository suggestConversionPriceRepository;

    public SuggestConversionPriceService(SuggestConversionPriceRepository suggestConversionPriceRepository) {
        this.suggestConversionPriceRepository = suggestConversionPriceRepository;
    }

    public void importTable(YtCluster sourceCluster, String sourceTablePath,
                            YtCluster targetCluster, String targetTablePath,
                            PpcProperty<LocalDate> lastSuccessfulUpdateDate,
                            LocalDate today) {
        YPath targetTableYpath = YPath.simple(targetTablePath);
        YPath sourceTableYpath = YPath.simple(sourceTablePath);
        if (isTableUnavailable(targetTableYpath, targetCluster) ||
                isTableUnavailable(sourceTableYpath, sourceCluster)) {
            return;
        }

        importTable(sourceCluster, sourceTablePath, targetCluster, targetTableYpath, lastSuccessfulUpdateDate, today);
    }

    private void importTable(YtCluster sourceCluster, String sourceTablePath,
                             YtCluster targetCluster, YPath targetTableYpath,
                             PpcProperty<LocalDate> lastSuccessfulUpdateDate,
                             LocalDate today) {
        logger.info("Start writing chunks from table {} in cluster {} to table {} in cluster {}",
                sourceTablePath, sourceCluster.getName(), targetTableYpath, targetCluster.getName());

        YtTable sourceTable = new YtTable(sourceTablePath);
        long tableSize = suggestConversionPriceRepository.getTableSize(sourceCluster, sourceTable);
        try {
            long chunkBegin = 0L;
            while (chunkBegin < tableSize) {
                long chunkEnd = min(chunkBegin + CHUNK_SIZE, tableSize);
                suggestConversionPriceRepository.importChunk(sourceCluster, sourceTable,
                        targetCluster, targetTableYpath,
                        chunkBegin, chunkEnd);
                chunkBegin += CHUNK_SIZE;
            }

            lastSuccessfulUpdateDate.set(today);
        } catch (IllegalStateException e) {
            logger.error(e.getMessage());
        }
    }

    private boolean isTableUnavailable(YPath tableYPath, YtCluster cluster) {
        return suggestConversionPriceRepository.isTableUnavailable(tableYPath, cluster);
    }

}
