package ru.yandex.direct.jobs.takeout;

import java.io.File;
import java.nio.charset.Charset;
import java.nio.file.Paths;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.mail.internet.MimeBodyPart;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import one.util.streamex.StreamEx;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.balance.client.BalanceClient;
import ru.yandex.direct.balance.client.model.request.FindClientRequest;
import ru.yandex.direct.balance.client.model.response.FindClientResponseItem;
import ru.yandex.direct.core.entity.addition.callout.container.CalloutSelection;
import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.service.CalloutService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.banner.model.BannerWithCallouts;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.BannerWithImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.Constants;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.model.AgencyClientRelation;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.AgencyClientRelationService;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.core.entity.image.service.ImageUtils;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterService;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService;
import ru.yandex.direct.core.entity.takeout.model.ProcessedData;
import ru.yandex.direct.core.entity.takeout.model.TakeoutJobParams;
import ru.yandex.direct.core.entity.takeout.model.TakeoutJobResult;
import ru.yandex.direct.core.entity.timetarget.service.GeoTimezoneMappingService;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.service.TurboLandingService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.service.VcardService;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.mail.EmailAddress;
import ru.yandex.direct.mail.MailMessage;
import ru.yandex.direct.mail.MailSender;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.rbac.RbacRepType;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.takeout.client.TakeoutClient;
import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.reader.FilterLogRecordsByCampaignTypeBuilder;
import ru.yandex.direct.useractionlog.reader.UserActionLogFilter;
import ru.yandex.direct.useractionlog.reader.UserActionLogOffset;
import ru.yandex.direct.useractionlog.reader.UserActionLogReader;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.io.TempDirectory;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.WEB_EDIT;
import static ru.yandex.direct.jobs.takeout.TakeoutRules.AVAILABLE_CAMPAIGN_OPTIONS;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Service
@ParametersAreNonnullByDefault
class TakeoutJobService {
    private static final Logger logger = LoggerFactory.getLogger(TakeoutUploadJob.class);

    private final TakeoutRules takeoutRules;
    private final TakeoutClient takeoutClient;
    private final ClientService clientService;
    private final UserService userService;
    private final CampaignService campaignService;
    private final BidModifierService bidModifierService;
    private final AdGroupService adGroupService;
    private final GeoTreeFactory geoTreeFactory;
    private final GeoBaseHelper geoBaseHelper;
    private final KeywordService keywordService;
    private final RelevanceMatchRepository relevanceMatchRepository;
    private final RetargetingService retargetingService;
    private final PerformanceFilterService performanceFilterService;
    private final BannerService bannerService;
    private final VcardService vcardService;
    private final SitelinkSetService sitelinkSetService;
    private final CalloutService calloutService;
    private final TurboLandingService turboLandingService;
    private final CreativeService creativeService;
    private final BannerImageFormatRepository bannerImageFormatRepository;
    private final MailSender mailSender;
    private final BalanceClient balanceClient;
    private final AgencyClientRelationService agencyClientRelationService;
    private final UserActionLogReader userActionLogReader;
    private final DbQueueRepository dbQueueRepository;
    private final EnvironmentType environmentType;
    private final FilterLogRecordsByCampaignTypeBuilder filterLogRecordsByCampaignTypeBuilder;
    private final GeoTimezoneMappingService geoTimezoneMappingService;
    private final FeedService feedService;
    private final ClientGeoService clientGeoService;

    public TakeoutJobService(TakeoutRules takeoutRules,
                             TakeoutClient takeoutClient,
                             ClientService clientService,
                             UserService userService,
                             CampaignService campaignService,
                             BidModifierService bidModifierService,
                             AdGroupService adGroupService,
                             GeoTreeFactory geoTreeFactory,
                             GeoBaseHelper geoBaseHelper,
                             KeywordService keywordService,
                             RelevanceMatchRepository relevanceMatchRepository,
                             RetargetingService retargetingService,
                             PerformanceFilterService performanceFilterService,
                             BannerService bannerService,
                             VcardService vcardService,
                             SitelinkSetService sitelinkSetService,
                             CalloutService calloutService,
                             TurboLandingService turboLandingService,
                             CreativeService creativeService,
                             BannerImageFormatRepository bannerImageFormatRepository,
                             MailSender mailSender,
                             BalanceClient balanceClient,
                             AgencyClientRelationService agencyClientRelationService,
                             UserActionLogReader userActionLogReader,
                             DbQueueRepository dbQueueRepository,
                             EnvironmentType environmentType,
                             FilterLogRecordsByCampaignTypeBuilder filterLogRecordsByCampaignTypeBuilder,
                             GeoTimezoneMappingService geoTimezoneMappingService, FeedService feedService,
                             ClientGeoService clientGeoService) {
        this.takeoutRules = takeoutRules;
        this.takeoutClient = takeoutClient;
        this.clientService = clientService;
        this.userService = userService;
        this.campaignService = campaignService;
        this.bidModifierService = bidModifierService;
        this.adGroupService = adGroupService;
        this.geoTreeFactory = geoTreeFactory;
        this.geoBaseHelper = geoBaseHelper;
        this.keywordService = keywordService;
        this.relevanceMatchRepository = relevanceMatchRepository;
        this.retargetingService = retargetingService;
        this.performanceFilterService = performanceFilterService;
        this.bannerService = bannerService;
        this.vcardService = vcardService;
        this.sitelinkSetService = sitelinkSetService;
        this.calloutService = calloutService;
        this.turboLandingService = turboLandingService;
        this.creativeService = creativeService;
        this.bannerImageFormatRepository = bannerImageFormatRepository;
        this.mailSender = mailSender;
        this.balanceClient = balanceClient;
        this.agencyClientRelationService = agencyClientRelationService;
        this.userActionLogReader = userActionLogReader;
        this.dbQueueRepository = dbQueueRepository;
        this.environmentType = environmentType;
        this.filterLogRecordsByCampaignTypeBuilder = filterLogRecordsByCampaignTypeBuilder;
        this.geoTimezoneMappingService = geoTimezoneMappingService;
        this.feedService = feedService;
        this.clientGeoService = clientGeoService;
    }

    private static final ObjectMapper MAPPER;

    static {
        MAPPER = JsonUtils.MAPPER.copy();
        MAPPER.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
        MAPPER.setSerializationInclusion(JsonInclude.Include.NON_NULL);
    }

    public TakeoutJobResult proceedJob(int shard, DbQueueJob<TakeoutJobParams, TakeoutJobResult> dbQueueJob) {
        TakeoutJobParams params = dbQueueJob.getArgs();
        ClientId clientId = dbQueueJob.getClientId();

        logger.info("uploading started for uid {} (ClientID {})", params.getUid(), clientId);
        ProcessedData processed = Optional.ofNullable(params.getProcessedData()).orElse(new ProcessedData());
        try (TempDirectory tempDirectory = new TempDirectory("takeout")) {
            User user = userService.getUser(params.getUid());
            logger.info("User data {}", user);
            //todo access логи по uid
            userData(params.getUid(), user, params.getJobId(), tempDirectory, processed);
            //по clientId выгружаем если uid есть в базе директа
            if (user != null && !user.getStatusBlocked()) {
                clientData(shard, user, params.getJobId(), tempDirectory, processed);
            }
        } finally {
            params.withProcessedData(processed);
            logger.debug("Job args updated, job {}", dbQueueJob.getId());
            dbQueueRepository.updateArgs(shard, dbQueueJob, params);
        }
        logger.info("upload done for job_id {}, {} files uploaded", params.getJobId(),
                processed.getUploadedFiles().size());
        //todo убрать для прода
        logger.info(JsonUtils.toJson(processed));
        var result = takeoutClient.done(processed.getUploadedFiles(), params.getJobId());
        logger.info("takeout done response: {}", result);
        if (!"ok".equals(result.getStatus())) {
            throw new TakeoutUploadJobException(String.format("done status is not ok: %s", result));
        }
        return TakeoutJobResult.success(processed);
    }

    /**
     * Данные по uid: пользлоги и настройки пользователя
     */
    private void userData(long uid, @Nullable User user, String jobId, TempDirectory tempDirectory,
                          ProcessedData processed) {
        //user settings
        if (!processed.getUserSettings()) {
            if (user != null) {
                Map<String, Object> userData = takeoutRules.exportModel(user);
                String userFileName = generateFilename(tempDirectory, "user_" + uid);
                processData(userFileName, userData, jobId, processed);
                logger.info("User settings uploaded");
            } else {
                logger.info("User not found");
            }
            processed.setUserSettings(true);
        }
    }

    /**
     * Пользлоги с поиском по uid
     */
    private void userActionLogData(long uid, ClientId clientId, String jobId, TempDirectory tempDirectory,
                                   ProcessedData processed) {
        int chunkSize = 100_000;
        //максимальное количество ошибок при запросах clickhouse
        //после превышения выгрузка user action logs скипается
        int tryCount = 10;
        int currentTry = 0;

        String token = processed.getUserActionLogsToken();
        UserActionLogOffset offset = token == null ? null : UserActionLogOffset.fromToken(token);
        UserActionLogReader.FilterResult userActionLogs;
        do {
            try {
                userActionLogs = userActionLogReader
                        .filterActionLog(
                                new UserActionLogFilter().withOperatorUids(singletonList(uid)).withInternal(true),
                                chunkSize, offset,
                                ReadActionLogTable.Order.ASC,
                                filterLogRecordsByCampaignTypeBuilder.builder().withClientId(clientId.asLong()).build());
                logger.info("{} user action logs found", userActionLogs.getRecords().size());
            } catch (Exception e) {
                logger.error("Exception while obtaining user action logs on try {}", currentTry++);
                tryCount--;
                continue;
            }
            offset = userActionLogs.getOffset();
            if (userActionLogs.getRecords().isEmpty()) {
                break;
            }
            String logsFileName = generateFilename(tempDirectory,
                    "user_action_logs" +
                            (offset == null ? ""
                                    : "_" + offset.getDateTime().toEpochSecond(ZoneOffset.UTC)));
            processData(logsFileName, userActionLogs.getRecords(), jobId, processed);
            if (offset == null) {
                break;
            } else {
                processed.setUserActionLogsToken(offset.toToken());
            }
        } while (tryCount > 0);

        if (tryCount == 0) {
            throw new TakeoutUploadJobException("Cannot obtain user action logs");
        }
        logger.info("User action logs uploaded");
    }

    private void clientData(int shard,
                            User user,
                            String jobId,
                            TempDirectory tempDirectory,
                            ProcessedData processed) {
        if (!processed.isRegionAccepted()) {
            if (!isEuropeanClient(user.getUid())) {
                logger.info("client region is not in Europe");
                return;
            } else {
                processed.setRegionAccepted(true);
            }
        }

        Client client = clientService.massGetClientsByUids(singletonList(user.getId())).get(user.getId());
        if (client == null) {
            logger.error("Client not found for uid {}. Something wrong with ppcdict data", user.getId());
            return;
        }

        ClientId clientId = ClientId.fromLong(client.getId());

        //user action logs
        if (!processed.getUserActionLogs()) {
            userActionLogData(user.getUid(), clientId, jobId, tempDirectory, processed);
            processed.setUserActionLogs(true);
        }

        if (client.getRole() == RbacRole.AGENCY) {
            logger.info("client {} is agency", client.getId());
            List<AgencyClientRelation> relations = agencyClientRelationService.getAgencyClients(clientId);
            logger.info("{} clients found", relations.size());
            for (AgencyClientRelation relation : relations) {
                if (processed.getClients().contains(relation.getClientClientId().asLong())) {
                    continue;
                }
                ClientId clientOfAgencyId = relation.getClientClientId();
                Client clientOfAgency = clientService.getClient(clientOfAgencyId);
                if (clientOfAgency == null) {
                    logger.warn("client {} not found but relation exists", clientOfAgencyId);
                    continue;
                }
                if (user.getRepType() == RbacRepType.LIMITED
                        && !user.getUid().equals(clientOfAgency.getAgencyUserId())) {
                    logger.info("Client {} skipped for limited agency rep", relation.getClientClientId());
                    continue;
                }
                List<Campaign> campaigns = campaignService
                        .searchNotEmptyCampaignsByClientIdAndTypes(relation.getClientClientId(), WEB_EDIT);
                logger.info("{} campaigns found for client {}", campaigns.size(), clientOfAgencyId);
                generateClientFiles(shard, clientOfAgency, jobId, campaigns, tempDirectory, processed);
                logger.info("{} campaigns of serviced client {} uploaded", campaigns.size(), clientId);
                processed.addClientId(clientId.asLong());
            }
        } else if (client.getRole() == RbacRole.CLIENT) {
            List<Campaign> campaigns = campaignService.searchNotEmptyCampaignsByClientIdAndTypes(clientId, WEB_EDIT);
            logger.info("{} campaigns found", campaigns.size());
            generateClientFiles(shard, client, jobId, campaigns, tempDirectory, processed);
        }
    }

    private boolean isEuropeanClient(long uid) {
        List<FindClientResponseItem> balanceInfo = balanceClient.findClient(new FindClientRequest().withUid(uid));
        FindClientResponseItem clientInfo = balanceInfo.stream().findFirst().orElse(null);
        if (clientInfo == null) {
            logger.info("client not found in balance for uid {}", uid);
            return false;
        }
        if (clientInfo.getRegionId() == null) {
            logger.info("client region is null for uid {}", uid);
            return false;
        }
        logger.info("client region for uid {} is {}", uid, clientInfo.getRegionId());
        return geoBaseHelper.getParentRegionIds(clientInfo.getRegionId())
                .contains((int) Region.EUROPE_REGION_ID);
    }

    private void generateClientFiles(int shard,
                                     Client client,
                                     String jobId,
                                     List<Campaign> campaigns,
                                     TempDirectory tempDirectory,
                                     ProcessedData processed) {
        ClientId clientId = ClientId.fromLong(client.getClientId());

        if (!processed.feeds()) {
            List<Feed> feeds = feedService.getFeeds(clientId, FeedQueryFilter.newBuilder().build());
            if (!feeds.isEmpty()) {
                Map<String, Object> feedsData = new HashMap<>();
                for (Feed feed : feeds) {
                    feedsData.put(feed.getId().toString(), takeoutRules.exportModel(feed));
                }
                String feedsFileName = generateFilename(tempDirectory, "feeds_" + clientId.toString());
                processData(feedsFileName, feedsData, jobId, processed);
            }
            processed.setFeeds(true);
        }

        if (campaigns.isEmpty()) {
            logger.info("No campaigns found for client {}", client.getId());
            return;
        }

        Set<Long> toDo = StreamEx.of(campaigns)
                .map(Campaign::getId)
                .filter(id -> !processed.getCampaignIds().contains(id))
                .toSet();

        Campaign campaign;
        Map<String, Object> campData;
        logger.info("{} campaigns to export", toDo.size());
        for (Long campaignId : toDo) {
            logger.info("starting export campaign {}", campaignId);
            try {
                campaign = campaignService.getCampaigns(clientId,
                        Collections.singletonList(campaignId)).get(0);
                campData = getCampaignData(campaign, client.getCountryRegionId());
                campData.put("adGroups", getGroupsData(shard, campaign, client.getCountryRegionId()));
                campData.put("banners", getBannersData(shard, campaign));
            } catch (Exception ex) {
                logger.error("exporting failed for campaign " + campaignId, ex);
                throw new TakeoutUploadJobException("Exception while processing campaign " + campaignId, ex);
            }
            String campFileName = generateFilename(tempDirectory, "campaign_" + campaign.getId().toString());
            processData(campFileName, campData, jobId, processed);
            logger.info("campaign {} uploaded", campaignId);
            processed.addCampaignId(campaignId);
        }
    }

    private String generateFilename(TempDirectory tempDirectory, String id) {
        return Paths.get(tempDirectory.getPath().toString(), id + ".json").toString();
    }

    private Map<String, Object> getBidModifiers(List<BidModifier> bidModifiers) {
        Map<String, Object> bidModifiersData = new HashMap<>();
        for (BidModifier bidModifier : bidModifiers) {
            bidModifiersData.put(bidModifier.getId().toString(), takeoutRules.exportModel(bidModifier));
        }
        return bidModifiersData;
    }

    Map<String, Object> getCampaignData(Campaign campaign, Long regionId) {
        ClientId clientId = ClientId.fromLong(campaign.getClientId());
        Map<String, Object> campaignData = takeoutRules.exportModel(campaign);
        if (campaign.getTimeTarget() != null) {
            campaignData.put(Campaign.TIME_TARGET.name(), campaign.getTimeTarget().toString());
        }
        List<BidModifier> bidModifiers = bidModifierService.getByCampaignIds(clientId, singletonList(campaign.getId()),
                Constants.ALL_TYPES,
                new HashSet<>(Collections.singletonList(BidModifierLevel.CAMPAIGN)), campaign.getUserId());
        if (!bidModifiers.isEmpty()) {
            Map<String, Object> bidModifiersData = getBidModifiers(bidModifiers);
            campaignData.put("bid_modifiers", bidModifiersData);
        }
        if (campaign.getGeo() != null) {
            campaignData.put("geo", performGeo(campaign.getGeo(), regionId));
        }
        if (campaign.getOpts() != null && !campaign.getOpts().isEmpty()) {
            campaignData.put("options", StreamEx.of(campaign.getOpts())
                    .filter(AVAILABLE_CAMPAIGN_OPTIONS::contains)
                    .toSet());
        }
        if (campaign.getTimezoneId() != null) {
            campaignData.put("timezone",
                    geoTimezoneMappingService.getRegionIdByTimezoneId(campaign.getTimezoneId()).getNameEn());
        }
        return campaignData;
    }

    Map<String, Object> getGroupsData(int shard, Campaign campaign, Long regionId) {
        ClientId clientId = ClientId.fromLong(campaign.getClientId());
        List<Long> adGroupIds = adGroupService.getAdGroupIdsByCampaignIds(Collections.singleton(campaign.getId()))
                .getOrDefault(campaign.getId(), emptyList());
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }

        List<AdGroup> adGroups = adGroupService.getAdGroups(clientId, adGroupIds);
        adGroupService.enrichAdGroupsWithEffectiveAndRestrictedGeo(shard, adGroups);

        Map<Long, List<Keyword>> keywordMap =
                keywordService.getKeywordsByAdGroupIds(clientId, adGroupIds);
        Map<Long, List<TargetInterest>> targetInterestMap =
                StreamEx.of(retargetingService.getTargetInterestsWithInterestByAdGroupIds(adGroupIds, clientId, shard))
                        .groupingBy(TargetInterest::getAdGroupId);

        Map<Long, RelevanceMatch> relevanceMatchMap =
                relevanceMatchRepository.getRelevanceMatchesByAdGroupIds(shard, clientId, adGroupIds);

        Map<Long, List<PerformanceFilter>> performanceFiltersMap =
                performanceFilterService.getPerformanceFilters(clientId, new ArrayList<>(adGroupIds));

        Map<Long, List<BidModifier>> adGroupsBidModifiersMap =
                StreamEx.of(bidModifierService.getByAdGroupIds(shard, new HashSet<>(adGroupIds),
                        Constants.ALL_TYPES, new HashSet<>(Collections.singletonList(BidModifierLevel.ADGROUP))))
                        .groupingBy(BidModifier::getAdGroupId);

        Map<String, Object> groupsData = new HashMap<>();

        for (AdGroup adGroup : adGroups) {
            Map<String, Object> groupData = takeoutRules.exportModel(adGroup);
            if (keywordMap.containsKey(adGroup.getId())) {
                Map<String, Object> keywordData = new HashMap<>();
                for (Keyword keyword : keywordMap.get(adGroup.getId())) {
                    keywordData.put(keyword.getId().toString(), takeoutRules.exportModel(keyword));
                }
                groupData.put("keywords", keywordData);
            }
            if (targetInterestMap.containsKey(adGroup.getId())) {
                Map<String, Object> targetInterestData = new HashMap<>();
                for (TargetInterest targetInterest : targetInterestMap.get(adGroup.getId())) {
                    targetInterestData.put(targetInterest.getId().toString(), takeoutRules.exportModel(targetInterest));
                }
                groupData.put("retargetings", targetInterestData);
            }
            if (relevanceMatchMap.containsKey(adGroup.getId())) {
                groupData.put("autotargeting", takeoutRules.exportModel(relevanceMatchMap.get(adGroup.getId())));
            }
            if (performanceFiltersMap.containsKey(adGroup.getId())) {
                Map<String, Object> performanceFiltersData = new HashMap<>();
                for (PerformanceFilter performanceFilter : performanceFiltersMap.get(adGroup.getId())) {
                    Map<String, Object> performanceFilterData = takeoutRules.exportModel(performanceFilter);
                    performanceFiltersData.put(performanceFilter.getId().toString(), performanceFilterData);
                }
                groupData.put("smart_filters", performanceFiltersData);
            }
            if (adGroupsBidModifiersMap.containsKey(adGroup.getId())) {
                Map<String, Object> bidModifiersData = getBidModifiers(adGroupsBidModifiersMap.get(adGroup.getId()));
                if (!bidModifiersData.isEmpty()) {
                    groupData.put("bid_modifiers", bidModifiersData);
                }
            }
            if (adGroup.getGeo() != null) {
                groupData.put("geo", performGeo(adGroup.getGeo(), regionId));
            }
            // обрабатывается поле effectiveGeo после вызова enrichAdGroupsWithEffectiveAndRestrictedGeo(), так как там
            // используемое геодерево теперь всегда типа GeoTreeType.API.
            if (adGroup.getEffectiveGeo() != null) {
                adGroup.setEffectiveGeo(clientGeoService.convertForWeb(adGroup.getEffectiveGeo(),
                        geoTreeFactory.getTranslocalGeoTree(regionId)));
            }
            groupsData.put(adGroup.getId().toString(), groupData);
        }
        return groupsData;
    }

    Map<String, Object> getBannersData(int shard, Campaign campaign) {
        ClientId clientId = ClientId.fromLong(campaign.getClientId());
        Map<String, Object> bannersData = new HashMap<>();
        List<BannerWithSystemFields> banners =
                bannerService.getBannersByCampaignIds(Collections.singletonList(campaign.getId()));

        Set<Long> vcardIds = StreamEx.of(banners)
                .select(BannerWithVcard.class)
                .map(BannerWithVcard::getVcardId)
                .nonNull()
                .toSet();
        Map<Long, Vcard> vcardMap = vcardService.getVcardsById(ClientId.fromLong(campaign.getClientId()), vcardIds);

        Set<Long> siteLinkSetIds = StreamEx.of(banners)
                .select(BannerWithSitelinks.class)
                .map(BannerWithSitelinks::getSitelinksSetId)
                .nonNull()
                .toSet();

        Map<Long, SitelinkSet> sitelinkSetMap = listToMap(sitelinkSetService
                .getSitelinkSets(siteLinkSetIds, ClientId.fromLong(campaign.getClientId()),
                        LimitOffset.limited(siteLinkSetIds.size())), SitelinkSet::getId);

        Set<Long> calloutIds = StreamEx.of(banners)
                .select(BannerWithCallouts.class)
                .flatCollection(BannerWithCallouts::getCalloutIds)
                .nonNull()
                .toSet();
        Map<Long, Callout> calloutMap = listToMap(calloutService
                .getCallouts(clientId, new CalloutSelection()
                                .withIds(new ArrayList<>(calloutIds))
                                .withDeleted(false),
                        LimitOffset.limited(calloutIds.size())), Callout::getId);

        Set<Long> turboLandingIds = StreamEx.of(banners)
                .select(BannerWithTurboLanding.class)
                .map(BannerWithTurboLanding::getTurboLandingId)
                .nonNull()
                .toSet();
        turboLandingIds.addAll(StreamEx.of(sitelinkSetMap.values())
                .flatCollection(SitelinkSet::getSitelinks)
                .nonNull()
                .map(Sitelink::getTurboLandingId)
                .nonNull()
                .toSet());
        Map<Long, TurboLanding> turboLandingMap =
                listToMap(turboLandingService
                                .getClientTurboLandingsById(clientId, new ArrayList<>(turboLandingIds),
                                        LimitOffset.maxLimited()),
                        ModelWithId::getId);

        Set<Long> creativeIds = StreamEx.of(banners)
                .select(BannerWithCreative.class)
                .map(BannerWithCreative::getCreativeId)
                .toSet();
        Map<Long, Creative> creativeMap = listToMap(creativeService.get(clientId, new ArrayList<>(creativeIds), null),
                Creative::getId);

        Set<String> imageHashes = StreamEx.of(banners)
                .select(BannerWithImage.class)
                .map(BannerWithImage::getImageHash)
                .nonNull()
                .toSet();

        imageHashes.addAll(
                StreamEx.of(banners)
                        .select(TextBanner.class)
                        .map(TextBanner::getImageHash)
                        .nonNull()
                        .toSet()
        );

        Map<String, BannerImageFormat> bannerImageFormats =
                bannerImageFormatRepository.getBannerImageFormats(shard, imageHashes);

        for (BannerWithSystemFields banner : banners) {
            Map<String, Object> bannerData = takeoutRules.exportModel(banner);
            if (bannerData.isEmpty()) {
                continue;
            }
            if (banner instanceof BannerWithVcard) {
                Vcard vcard = vcardMap.get(((BannerWithVcard) banner).getVcardId());
                if (vcard != null) {
                    bannerData.put("vcard", takeoutRules.exportModel(vcard));
                }
            }
            if (banner instanceof BannerWithSitelinks) {
                SitelinkSet sitelinkSet = sitelinkSetMap.get(((BannerWithSitelinks) banner).getSitelinksSetId());
                if (sitelinkSet != null) {
                    Map<String, Object> sitelinkSetData = new HashMap<>();
                    for (Sitelink sitelink : sitelinkSet.getSitelinks()) {
                        Map<String, Object> sitelinkData = takeoutRules.exportModel(sitelink);
                        if (sitelink.getTurboLandingId() != null) {
                            sitelinkData.put("turbolanding",
                                    takeoutRules.exportModel(turboLandingMap.get(sitelink.getTurboLandingId())));
                        }
                        sitelinkSetData.put(sitelink.getId().toString(), sitelinkData);
                    }
                    bannerData.put("sitelinks", sitelinkSetData);
                }
            }
            if (banner instanceof BannerWithCallouts) {
                Map<String, Object> calloutData = new HashMap<>();
                BannerWithCallouts bannerWithCallouts = (BannerWithCallouts) banner;
                if (!bannerWithCallouts.getCalloutIds().isEmpty()) {
                    for (Long calloutId : ((BannerWithCallouts) banner).getCalloutIds()) {
                        calloutData.put(calloutId.toString(), takeoutRules.exportModel(calloutMap.get(calloutId)));
                    }
                    bannerData.put("callouts", calloutData);
                }
            }
            if (banner instanceof BannerWithTurboLanding) {
                Long turboLandingId = ((BannerWithTurboLanding) banner).getTurboLandingId();
                if (turboLandingId != null) {
                    bannerData.put("turbolanding", takeoutRules.exportModel(turboLandingMap.get(turboLandingId)));
                }
            }
            if (banner instanceof BannerWithCreative) {
                Long creativeId = ((BannerWithCreative) banner).getCreativeId();
                if (creativeId != null) {
                    bannerData.put("creative", takeoutRules.exportModel(creativeMap.get(creativeId)));
                }
            }
            if (banner instanceof BannerWithImage) {
                String imageHash = ((BannerWithImage) banner).getImageHash();
                if (imageHash != null) {
                    BannerImageFormat format = bannerImageFormats.get(imageHash);
                    if (format != null) {
                        bannerData.put("image", getUrl(format));
                    }
                }
            }
            if (banner instanceof TextBanner) {
                String imageHash = ((TextBanner) banner).getImageHash();
                if (imageHash != null) {
                    BannerImageFormat format = bannerImageFormats.get(imageHash);
                    if (format != null) {
                        bannerData.put("banner_image", getUrl(format));
                    }
                }
            }
            bannersData.put(banner.getId().toString(), bannerData);
        }
        return bannersData;
    }

    //get-direct-picture/995451/--5CXgwyWEJvv4PZFv_NQA/orig
    private String getUrl(BannerImageFormat format) {
        return ImageUtils.generateOrigImageUrl(format);
    }

    private void processData(String filename, Object data, String jobId, ProcessedData processed) {
        try {
            logger.debug("Uploading file {}", filename);
            String stringData = MAPPER.writeValueAsString(data);
            File file = new File(filename);
            FileUtils.write(file, stringData, Charset.forName("UTF-8"));
            logger.debug("{} bites written to file {}", stringData.chars().count(), file.getName());
            var result = takeoutClient.uploadFile(file, jobId);
            if(!"ok".equals(result.getStatus())) {
                throw new TakeoutUploadJobException(String.format("upload status is not ok: %s", result));
            }
            logger.info("File uploaded: {}", filename);
            processed.addUploadedFile(file);
            if (environmentType.equals(EnvironmentType.TESTING)) {
                sendMail(jobId, file);
            }
            file.delete();
        } catch (Exception e) {
            logger.error("cannot process file " + filename, e);
            throw new TakeoutUploadJobException("Cannot save user settings data", e);
        }
    }

    private <T extends Number> List<List<String>> performGeo(@Nullable Collection<T> geo, Long clientRegionId) {
        if (geo == null) {
            return null;
        }
        GeoTree geoTree = geoTreeFactory.getTranslocalGeoTree(clientRegionId);
        return StreamEx.of(geo)
                .nonNull()
                .map(id -> {
                    List<String> geoList = new ArrayList<>();
                    Region region = geoTree.getRegion(Math.abs(NumberUtils.toLong(String.valueOf(id))));
                    geoList.add(id.toString());
                    if (region != null) {
                        geoList.add(region.getNameEn());
                    } else {
                        logger.warn("Unknown region: {}", id);
                    }
                    return geoList;
                })
                .toList();
    }

    /**
     * Метод для тестирования
     */
    private void sendMail(String jobId, File attachment) {
        String subject = String.format("Takeout %s %s", attachment.getName(), jobId);
        MimeBodyPart bodyPart = new MimeBodyPart();
        try {
            bodyPart.attachFile(attachment);
        } catch (Exception e) {
            logger.warn("cannot attach file to email", e);
        }
        MailMessage mailMessage =
                new MailMessage(new EmailAddress("direct-takeout@yandex.ru", "direct-takeout"),
                        new EmailAddress("direct-takeout-test@yandex-team.ru", "direct-takeout"), subject, "",
                        MailMessage.EmailContentType.TEXT, singletonList(bodyPart));
        mailSender.send(mailMessage);
    }
}
