package ru.yandex.direct.jobs.takeout;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.takeout.model.TakeoutJobParams;
import ru.yandex.direct.core.entity.takeout.model.TakeoutJobResult;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.service.DbQueueService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.utils.ThreadUtils;

import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.TAKEOUT_REQUEST;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_KUHTICH,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        ),
        tags = {DIRECT_PRIORITY_1})
@Hourglass(periodInSeconds = 60)
@ParametersAreNonnullByDefault
public class TakeoutUploadJob extends DirectShardedJob {
    private static final int JOB_RETRIES = 10;
    private static final int RETRY_TIMEOUT = 10;
    private static final Duration GRABBED_TIME = Duration.ofHours(15);

    private static final Logger logger = LoggerFactory.getLogger(TakeoutUploadJob.class);

    private final DbQueueService dbQueueService;
    private final TakeoutJobService takeoutService;
    private final int jobRetries;

    @Autowired
    public TakeoutUploadJob(DbQueueService dbQueueService, TakeoutJobService takeoutService) {
        this.dbQueueService = dbQueueService;
        this.takeoutService = takeoutService;
        jobRetries = JOB_RETRIES;
    }

    /**
     * Конструктор нужен только для тестов. Используется для указания шарда.
     */
    TakeoutUploadJob(int shard, DbQueueService dbQueueService, TakeoutJobService takeoutService, int jobRetries) {
        super(shard);
        this.dbQueueService = dbQueueService;
        this.takeoutService = takeoutService;
        this.jobRetries = jobRetries;
    }

    @Override
    public void execute() {
        while (dbQueueService
                .grabAndProcessJob(getShard(), TAKEOUT_REQUEST, GRABBED_TIME, this::processGrabbedJob, jobRetries,
                        this::onError)) {
            ThreadUtils.sleep(Duration.ofSeconds(RETRY_TIMEOUT).toMillis());
        }
    }

    private TakeoutJobResult processGrabbedJob(DbQueueJob<TakeoutJobParams, TakeoutJobResult> jobInfo) {
        logger.info("Start processing jobId={}, attempt={}", jobInfo.getId(), jobInfo.getTryCount());
        return takeoutService.proceedJob(getShard(), jobInfo);
    }


    private TakeoutJobResult onError(DbQueueJob<TakeoutJobParams, TakeoutJobResult> jobInfo,
                                     String stacktrace) {
        return TakeoutJobResult.error(jobInfo.getArgs().getProcessedData(), stacktrace);
    }
}
