package ru.yandex.direct.jobs.telephony

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService.getPhoneIdsIfTelephonyPhoneChanged
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneType
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob
import java.time.LocalDateTime
import javax.annotation.ParametersAreNonnullByDefault

const val CHUNK_UPDATE_SIZE = 1000
const val DEFAULT_DAYS_TELEPHONY_CALC_LAST_SHOW_TIME_JOB_NOT_STARTED = 2L
const val DEFAULT_MAX_DAYS_WITHOUT_SHOWS_ADV = 14
const val DEFAULT_MAX_DAYS_WITHOUT_ACTIONS_SITE = 30

/**
 * Джоба для возвращения номеров Телефонии для коллтрекинга в ТГО и на сайте
 * Отрываются джобой [TelephonyDetacherJob]
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(hours = 3, minutes = 5),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_2, CheckTag.DIRECT_CALLTRACKING],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT])
    ]
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = ProductionOnly::class)
@ParametersAreNonnullByDefault
class TelephonyAttacherJob @Autowired constructor(
    private val ppcPropertiesSupport: PpcPropertiesSupport,
    private val clientPhoneRepository: ClientPhoneRepository,
    private val telephonyPhoneService: TelephonyPhoneService
) : DirectShardedJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(TelephonyAttacherJob::class.java)
    }

    /**
     * Используется для тестов, т.к. нужно явно указать номер шарда
     */
    constructor(
        shard: Int,
        ppcPropertiesSupport: PpcPropertiesSupport,
        clientPhoneRepository: ClientPhoneRepository,
        telephonyPhoneService: TelephonyPhoneService
    ) : this(ppcPropertiesSupport, clientPhoneRepository, telephonyPhoneService) {
        super.withShard(shard)
    }

    private val isJobEnabled: Boolean
        get() = ppcPropertiesSupport.get(PpcPropertyNames.TELEPHONY_ATTACHING_ENABLED).getOrDefault(false)

    private val maxDaysWithoutShowsAdv: Int
        get() = ppcPropertiesSupport.get(PpcPropertyNames.MAX_DAYS_WITHOUT_SHOWS_FOR_ADV_TELEPHONY_PHONE)
            .getOrDefault(DEFAULT_MAX_DAYS_WITHOUT_SHOWS_ADV)

    private val maxDaysWithoutActionsSite: Int
        get() = ppcPropertiesSupport.get(PpcPropertyNames.MAX_DAYS_WITHOUT_ACTIONS_FOR_SITE_TELEPHONY_PHONE)
            .getOrDefault(DEFAULT_MAX_DAYS_WITHOUT_ACTIONS_SITE)

    override fun execute() {
        if (!isJobEnabled) {
            logger.info("Skip processing. Job is not enabled")
            return
        }

        val now = LocalDateTime.now()

        val calcLastShowTimeJobLastTimeProperty = ppcPropertiesSupport
            .get(PpcPropertyNames.TELEPHONY_CALC_LAST_SHOW_TIME_JOB_LAST_TIME)

        if (!isActualLastShowTimeCalculated(calcLastShowTimeJobLastTimeProperty, now)) {
            setJugglerStatus(
                JugglerStatus.CRIT,
                "Skip processing. Actual last_show_time values are not calculated.")
            logger.info("Skip processing. Actual last_show_time values are not calculated.")
            return
        }
        val detachedPhones = telephonyPhoneService.getDetachedTelephonyPhones(shard)
        val detachedAdvPhones = detachedPhones.getOrDefault(TelephonyPhoneType.ADV, mutableListOf())
        val detachedSitePhones = detachedPhones.getOrDefault(TelephonyPhoneType.SITE, mutableListOf())
        attachAdv(now, detachedAdvPhones)
        attachSite(now, detachedSitePhones)
    }

    private fun attachAdv(now: LocalDateTime, detachedAdvPhones: List<ClientPhone>) {
        logger.info("Find ${detachedAdvPhones.size} detached adv phones")
        detachedAdvPhones
            .asSequence()
            .mapNotNull { phone -> tryAttach(phone, now, TelephonyPhoneType.ADV) }
            .chunked(CHUNK_UPDATE_SIZE)
            .forEach {
                clientPhoneRepository.update(shard, it)
                telephonyPhoneService.resetStatusBsSynced(shard, getPhoneIdsIfTelephonyPhoneChanged(it))
            }
    }

    private fun attachSite(now: LocalDateTime, detachedSitePhones: List<ClientPhone>) {
        logger.info("Find ${detachedSitePhones.size} detached on site phones")
        detachedSitePhones
            .asSequence()
            .mapNotNull { phone -> tryAttach(phone, now, TelephonyPhoneType.SITE) }
            .chunked(CHUNK_UPDATE_SIZE)
            .forEach { clientPhoneRepository.update(shard, it) }
    }

    /**
     * Попытаться вернуть номер Телефонии для [phone],
     * Для коллтрекинга в ТГО: если [last_show_time] телефона больше чем [now] - [maxDaysWithoutShowsAdv].
     * Для коллтрекинга на сайте: если [last_show_time] телефона больше чем [now] - [maxDaysWithoutActionsSite].
     * Для успешного возвращения номера вернутся изменения для заданного [phone], иначе `null`
     */
    private fun tryAttach(
        phone: ClientPhone,
        now: LocalDateTime,
        type: TelephonyPhoneType
    ): AppliedChanges<ClientPhone>? {
        val phoneId = phone.id

        val mc = ModelChanges(phoneId, ClientPhone::class.java)

        if (isTooEarlyToAttach(phone, now, type)) {
            logger.info("last_show_time is too early to attach phone to $phoneId")
            return null
        }
        logger.info("Try to attach phone to $phoneId")
        try {
            val phoneValue = telephonyPhoneService.attachTelephony(phone, false)
                ?: throw IllegalStateException("Failed to attach telephony phone")
            mc.process(phoneValue.telephonyPhone, ClientPhone.TELEPHONY_PHONE)
            mc.process(phoneValue.telephonyServiceId, ClientPhone.TELEPHONY_SERVICE_ID)
            logger.info("Finish attach phone to $phoneId")
            return mc.applyTo(phone)
        } catch (e: RuntimeException) {
            logger.error("Failed to attach phone to $phoneId", e)
        }
        return null
    }

    private fun isActualLastShowTimeCalculated(
        calcLastShowTimeJobLastTimeProperty: PpcProperty<LocalDateTime>?,
        now: LocalDateTime
    ): Boolean {
        val lastTime = calcLastShowTimeJobLastTimeProperty?.get() ?: return false
        return lastTime.plusDays(DEFAULT_DAYS_TELEPHONY_CALC_LAST_SHOW_TIME_JOB_NOT_STARTED).isAfter(now)
    }

    private fun isTooEarlyToAttach(phone: ClientPhone, now: LocalDateTime, type: TelephonyPhoneType): Boolean {
        return type == TelephonyPhoneType.ADV &&
            now.minusDays(maxDaysWithoutShowsAdv.toLong()).isAfter(phone.lastShowTime) ||
            type == TelephonyPhoneType.SITE &&
            now.minusDays(maxDaysWithoutActionsSite.toLong()).isAfter(phone.lastShowTime)
    }

}
