package ru.yandex.direct.jobs.telephony;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneType;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.metrika.repository.CalltrackingNumberCallsRepository;
import ru.yandex.direct.core.entity.metrika.repository.CalltrackingNumberClicksRepository;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба высчитывает last_show_time и обновляет его у телефонов Телефонии
 * для коллтрекинга в ТГО и на сайте
 */
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 3, minutes = 5),
        needCheck = ProductionOnly.class,
        tags = CheckTag.DIRECT_CALLTRACKING,
        notifications = @OnChangeNotification(
                recipient = {NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING},
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class TelephonyCalcLastShowTimeJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(TelephonyCalcLastShowTimeJob.class);
    private static final Integer DEFAULT_MAX_DAYS_WITHOUT_ACTIONS = 30;
    private static final int CHUNK_UPDATE_SIZE = 1000;

    private final ClientPhoneRepository clientPhoneRepository;
    private final TelephonyPhoneService telephonyPhoneService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final ShardHelper shardHelper;
    private final CalltrackingNumberClicksRepository calltrackingNumberClicksRepository;
    private final CalltrackingNumberCallsRepository calltrackingNumberCallsRepository;

    @Autowired
    public TelephonyCalcLastShowTimeJob(
            ClientPhoneRepository clientPhoneRepository,
            TelephonyPhoneService telephonyPhoneService,
            PpcPropertiesSupport ppcPropertiesSupport,
            ShardHelper shardHelper,
            CalltrackingNumberClicksRepository calltrackingNumberClicksRepository,
            CalltrackingNumberCallsRepository calltrackingNumberCallsRepository) {
        this.clientPhoneRepository = clientPhoneRepository;
        this.telephonyPhoneService = telephonyPhoneService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.shardHelper = shardHelper;
        this.calltrackingNumberClicksRepository = calltrackingNumberClicksRepository;
        this.calltrackingNumberCallsRepository = calltrackingNumberCallsRepository;
    }

    @Override
    public void execute() {
        boolean isJobEnabled = ppcPropertiesSupport.get(PpcPropertyNames.TELEPHONY_CALC_LAST_SHOW_TIME_ENABLED)
                .getOrDefault(false);
        if (!isJobEnabled) {
            logger.info("Skip processing. Job is not enabled");
            return;
        }
        LocalDateTime now = LocalDateTime.now();
        Map<Integer, Map<TelephonyPhoneType, List<ClientPhone>>> phonesByShard = new HashMap<>();
        shardHelper.forEachShard(shard -> {
            Map<TelephonyPhoneType, List<ClientPhone>> phones = telephonyPhoneService.getTelephonyPhones(shard);
            phonesByShard.put(shard, phones);
        });
        calcLastShowTime(now, phonesByShard);
        ppcPropertiesSupport.get(PpcPropertyNames.TELEPHONY_CALC_LAST_SHOW_TIME_JOB_LAST_TIME).set(now);
    }

    private void calcLastShowTime(
            LocalDateTime now,
            Map<Integer, Map<TelephonyPhoneType, List<ClientPhone>>> phonesByShard
    ) {
        var sitePhonesContainer = getSiteTelephonyPhonesContainer(now, phonesByShard);
        shardHelper.forEachShard(shard -> {
            Map<ClientPhone, LocalDateTime> lastShowTimeByPhones = new HashMap<>();

            var advPhones = phonesByShard.getOrDefault(shard, Map.of()).getOrDefault(TelephonyPhoneType.ADV, List.of());
            var lastShowTimeByAdvPhones = telephonyPhoneService.getLastShowTimesByPhone(shard, advPhones);
            lastShowTimeByPhones.putAll(lastShowTimeByAdvPhones);

            var sitePhones = sitePhonesContainer.phonesByShard.get(shard);
            var lastShowTimeBySitePhones = getLastShowTimesBySitePhoneByShard(sitePhones, sitePhonesContainer);
            lastShowTimeByPhones.putAll(lastShowTimeBySitePhones);

            updateLastShowTime(shard, lastShowTimeByPhones);
        });
    }

    private Map<ClientPhone, LocalDateTime> getLastShowTimesBySitePhoneByShard(
            List<ClientPhone> sitePhones,
            SiteTelephonyPhonesContainer sitePhonesContainer
    ) {
        return listToMap(
                sitePhones,
                Function.identity(),
                p -> calcLastShowTime(
                        p,
                        sitePhonesContainer.lastCallTimesByPhones,
                        sitePhonesContainer.lastClickTimesByPhones
                )
        );
    }

    private LocalDateTime calcLastShowTime(
            ClientPhone phone,
            Map<String, LocalDateTime> lastCallTimesByPhones,
            Map<String, LocalDateTime> lastClickTimesByPhones
    ) {
        if (phone.getTelephonyPhone() != null) {
            LocalDateTime lastCallTime = lastCallTimesByPhones.get(phone.getTelephonyPhone().getPhone());
            LocalDateTime lastClickTime = lastClickTimesByPhones.get(phone.getTelephonyPhone().getPhone());
            return getLastShowTime(lastCallTime, lastClickTime, phone.getLastShowTime());
        }
        LocalDateTime lastClickTime = lastClickTimesByPhones.get(phone.getPhoneNumber().getPhone());
        return getLastShowTime(lastClickTime, phone.getLastShowTime());
    }

    /**
     * Получить актуальный last_show_time
     * Если еще не было звонков и кликов, то будем сравнивать с lastShowTime телефона
     */
    private LocalDateTime getLastShowTime(LocalDateTime... times) {
        LocalDateTime lastShowTime = null;
        for (LocalDateTime time : times) {
            if (lastShowTime == null && time != null) {
                lastShowTime = time;
                continue;
            }
            if (lastShowTime != null && time != null && time.isAfter(lastShowTime)) {
                lastShowTime = time;
            }
        }
        return lastShowTime;
    }

    private void updateLastShowTime(
            int shard,
            Map<ClientPhone, LocalDateTime> lastShowTimeByPhones
    ) {
        List<AppliedChanges<ClientPhone>> appliedChanges = EntryStream.of(lastShowTimeByPhones)
                .filterValues(Objects::nonNull)
                .map(e -> updateLastShowTime(e.getValue(), e.getKey()))
                .collect(Collectors.toList());
        List<Long> updatedLastShowTimePhoneIds = appliedChanges.stream()
                .filter(ac -> ac.changed(ClientPhone.LAST_SHOW_TIME))
                .map(ac -> ac.getModel().getId())
                .collect(Collectors.toList());
        Iterables.partition(appliedChanges, CHUNK_UPDATE_SIZE)
                .forEach(chunk -> clientPhoneRepository.update(shard, chunk));
        logger.info("Shard {}: update last_show_time for phones: {}", shard, updatedLastShowTimePhoneIds);
    }

    /**
     * Обновляет last_show_time для телефона
     */
    private AppliedChanges<ClientPhone> updateLastShowTime(LocalDateTime lastShowTime, ClientPhone phone) {
        Long phoneId = phone.getId();
        ModelChanges<ClientPhone> mc = new ModelChanges<>(phoneId, ClientPhone.class);
        mc.process(lastShowTime, ClientPhone.LAST_SHOW_TIME);
        return mc.applyTo(phone);
    }

    private SiteTelephonyPhonesContainer getSiteTelephonyPhonesContainer(
            LocalDateTime now,
            Map<Integer, Map<TelephonyPhoneType, List<ClientPhone>>> phonesByShard
    ) {
        Map<Integer, List<ClientPhone>> sitePhonesByShard = new HashMap<>();
        List<Long> counterIds = new ArrayList<>();
        List<String> phones = new ArrayList<>();
        List<String> telephonyPhones = new ArrayList<>();
        shardHelper.forEachShard(shard -> {
            var sitePhones =
                    phonesByShard.getOrDefault(shard, Map.of()).getOrDefault(TelephonyPhoneType.SITE, List.of());
            sitePhonesByShard.put(shard, sitePhones);
            counterIds.addAll(mapList(sitePhones, ClientPhone::getCounterId));

            telephonyPhones.addAll(sitePhones.stream()
                    .filter(p -> p.getTelephonyPhone() != null)
                    .map(p -> p.getTelephonyPhone().getPhone())
                    .collect(Collectors.toList()));
            phones.addAll(sitePhones.stream()
                    .filter(p -> p.getTelephonyPhone() == null)
                    .map(p -> p.getPhoneNumber().getPhone())
                    .collect(Collectors.toList()));
        });
        Map<String, LocalDateTime> lastCallTimesByPhones = getLastCallTimesByPhones(now, telephonyPhones);
        List<String> allPhones = new ArrayList<>();
        allPhones.addAll(telephonyPhones);
        allPhones.addAll(phones);
        var lastClickTimesByPhones =
                calltrackingNumberClicksRepository.getLastClickTimesByPhones(counterIds, allPhones);
        return new SiteTelephonyPhonesContainer(sitePhonesByShard, lastCallTimesByPhones, lastClickTimesByPhones);
    }

    private Map<String, LocalDateTime> getLastCallTimesByPhones(LocalDateTime now, List<String> phones) {
        Integer maxDaysWithoutActions = ppcPropertiesSupport
                .get(PpcPropertyNames.MAX_DAYS_WITHOUT_ACTIONS_FOR_SITE_TELEPHONY_PHONE)
                .getOrDefault(DEFAULT_MAX_DAYS_WITHOUT_ACTIONS);
        LocalDateTime nowMinusMaxDaysWithoutActions = now.minusDays(maxDaysWithoutActions);
        return calltrackingNumberCallsRepository.getLastCallTimesByPhones(now, nowMinusMaxDaysWithoutActions, phones);
    }

    private static class SiteTelephonyPhonesContainer {
        private final Map<Integer, List<ClientPhone>> phonesByShard;
        private final Map<String, LocalDateTime> lastCallTimesByPhones;
        private final Map<String, LocalDateTime> lastClickTimesByPhones;

        private SiteTelephonyPhonesContainer(
                Map<Integer, List<ClientPhone>> phonesByShard,
                Map<String, LocalDateTime> lastCallTimesByPhones,
                Map<String, LocalDateTime> lastClickTimesByPhones
        ) {
            this.phonesByShard = phonesByShard;
            this.lastCallTimesByPhones = lastCallTimesByPhones;
            this.lastClickTimesByPhones = lastClickTimesByPhones;
        }
    }

}
