package ru.yandex.direct.jobs.telephony;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneState;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба отвязывает номера Телефонии для коллтрекинга на сайте,
 * у которых установлен флаг is_deleted = 1
 * и last_show_time равно или меньше чем [текущее время - 4 часа]
 */
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 3, minutes = 5),
        tags = CheckTag.DIRECT_CALLTRACKING,
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = {NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING},
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 3, minutes = 5),
        tags = CheckTag.DIRECT_CALLTRACKING,
        needCheck = NonProductionEnvironment.class,
        notifications = @OnChangeNotification(
                recipient = {NotificationRecipient.LOGIN_MAXLOG},
                method = NotificationMethod.EMAIL,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class TelephonyDeletedPhonesDetacherJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(TelephonyDeletedPhonesDetacherJob.class);
    private static final Duration UNLINK_DELAY = Duration.ofHours(4);

    private final ClientPhoneRepository clientPhoneRepository;
    private final TelephonyPhoneService telephonyPhoneService;

    @Autowired
    public TelephonyDeletedPhonesDetacherJob(ClientPhoneRepository clientPhoneRepository,
                                             TelephonyPhoneService telephonyPhoneService) {
        this.clientPhoneRepository = clientPhoneRepository;
        this.telephonyPhoneService = telephonyPhoneService;
    }

    /**
     * Конструктор нужен только для тестов. Используется для указания шарда.
     */
    public TelephonyDeletedPhonesDetacherJob(
            int shard,
            ClientPhoneRepository clientPhoneRepository,
            TelephonyPhoneService telephonyPhoneService) {
        super(shard);
        this.clientPhoneRepository = clientPhoneRepository;
        this.telephonyPhoneService = telephonyPhoneService;
    }

    @Override
    public void execute() {
        int shard = getShard();
        LocalDateTime latestLastShowTime = LocalDateTime.now().minus(UNLINK_DELAY);
        Map<TelephonyPhoneState, List<ClientPhone>> phones =
                telephonyPhoneService.getDeletedTelephonyPhones(shard, latestLastShowTime);

        List<ClientPhone> readyToDeletePhones = phones.getOrDefault(TelephonyPhoneState.DETACHED, new ArrayList<>());
        if (!phones.containsKey(TelephonyPhoneState.ATTACHED)) {
            logger.info("No phones find to detach");
        } else {
            List<ClientPhone> attachedPhones = phones.get(TelephonyPhoneState.ATTACHED);
            logger.info("Find {} attached phones to detach", attachedPhones.size());
            List<ClientPhone> detachedPhones = detachPhones(attachedPhones);
            readyToDeletePhones.addAll(detachedPhones);
            logger.info("Finish detaching phones");
        }
        if (readyToDeletePhones.isEmpty()) {
            logger.info("No phones find to delete");
        } else {
            deletePhone(shard, readyToDeletePhones);
        }
    }

    /**
     * Метод возвращает номера, которые удалось успешно отвязать
     */
    private List<ClientPhone> detachPhones(List<ClientPhone> attachedPhones) {
        List<ClientPhone> detachedPhones = new ArrayList<>();
        for (ClientPhone phone : attachedPhones) {
            logger.info("Try to detach phone to {}", phone.getId());
            try {
                telephonyPhoneService.detachTelephony(phone.getTelephonyServiceId());
                logger.info("Finish detach phone to {}", phone.getId());
                detachedPhones.add(phone);
            } catch (RuntimeException ex) {
                logger.error("Failed to detach phone to {}", phone.getId(), ex);
            }
        }
        return detachedPhones;
    }

    private void deletePhone(int shard, List<ClientPhone> deletingPhones) {
        List<Long> clientIds = mapList(deletingPhones, p -> p.getClientId().asLong());
        List<Long> phoneIds = mapList(deletingPhones, ClientPhone::getId);
        logger.info("Try to delete {} phones", deletingPhones.size());
        clientPhoneRepository.delete(shard, clientIds, phoneIds);
        logger.info("Finish deleting {} phones", deletingPhones.size());
    }

}
