package ru.yandex.direct.jobs.telephony

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.*
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService.getPhoneIdsIfTelephonyPhoneChanged
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneType
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob
import java.time.LocalDateTime
import javax.annotation.ParametersAreNonnullByDefault


const val TELEPHONY_DETACHER_JOB_LAST_TIME_PROPERTY_PREFIX = "telephony_detacher_job_last_time_"

/**
 * Джоба для отрыва номеров Телефонии для коллтрекинга в ТГО и на сайте
 * Возвращаются джобой [TelephonyAttacherJob]
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 2 * 60 + 5), // два запуска и 5 минут, чтобы избежать флапов
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_2, CheckTag.DIRECT_CALLTRACKING],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT])]
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = ProductionOnly::class)
@ParametersAreNonnullByDefault
open class TelephonyDetacherJob @Autowired constructor(
    private val ppcPropertiesSupport: PpcPropertiesSupport,
    private val clientPhoneRepository: ClientPhoneRepository,
    private val telephonyPhoneService: TelephonyPhoneService
) : DirectShardedJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(TelephonyDetacherJob::class.java)
    }

    /**
     * Используется для тестов, т.к. нужно явно указать номер шарда
     */
    constructor(
        shard: Int,
        ppcPropertiesSupport: PpcPropertiesSupport,
        clientPhoneRepository: ClientPhoneRepository,
        telephonyPhoneService: TelephonyPhoneService
    ) : this(ppcPropertiesSupport, clientPhoneRepository, telephonyPhoneService) {
        super.withShard(shard)
    }

    private val isJobEnabled: Boolean
        get() = ppcPropertiesSupport.get(PpcPropertyNames.TELEPHONY_DETACHING_ENABLED).getOrDefault(false)

    private val maxDaysWithoutShowsAdv: Int
        get() = ppcPropertiesSupport.get(PpcPropertyNames.MAX_DAYS_WITHOUT_SHOWS_FOR_ADV_TELEPHONY_PHONE)
            .getOrDefault(DEFAULT_MAX_DAYS_WITHOUT_SHOWS_ADV)

    private val maxDaysWithoutActionsOnSite: Int
        get() = ppcPropertiesSupport.get(PpcPropertyNames.MAX_DAYS_WITHOUT_ACTIONS_FOR_SITE_TELEPHONY_PHONE)
            .getOrDefault(DEFAULT_MAX_DAYS_WITHOUT_ACTIONS_SITE)

    override fun execute() {
        if (!isJobEnabled) {
            logger.info("Skip processing. Job is not enabled")
            return
        }
        val now = LocalDateTime.now()

        val calcLastShowTimeJobLastTimeProperty = ppcPropertiesSupport
            .get(PpcPropertyNames.TELEPHONY_CALC_LAST_SHOW_TIME_JOB_LAST_TIME)

        if (!isActualLastShowTimeCalculated(calcLastShowTimeJobLastTimeProperty, now)) {
            setJugglerStatus(
                JugglerStatus.CRIT,
                "Skip processing. Actual last_show_time values are not calculated.")
            logger.info("Skip processing. Actual last_show_time values are not calculated.")
            return
        }

        // Необходимо, чтобы джоба успешно выполнялась раз в сутки. Запускаем ее каждый час
        // и записываем время последнего запуска в свойство [PpcPropertyNames.TELEPHONY_DETACHER_JOB_LAST_TIME_$shard]
        // Каждый час при запуске проверяем, прошли ли с момента успешного запуска одни сутки
        val lastTimePropertyName = PpcPropertyName(
            TELEPHONY_DETACHER_JOB_LAST_TIME_PROPERTY_PREFIX + shard,
            PpcPropertyType.LOCAL_DATE_TIME)
        val lastTimeProperty = ppcPropertiesSupport.get(lastTimePropertyName)
        if (!isTimeToExecute(lastTimeProperty, now)) {
            logger.info("Skip processing. Too early to execute job")
            return
        }
        val attachedPhones = telephonyPhoneService.getAttachedTelephonyPhones(shard)
        val attachedAdvPhones = attachedPhones.getOrDefault(TelephonyPhoneType.ADV, mutableListOf())
        val attachedSitePhones = attachedPhones.getOrDefault(TelephonyPhoneType.SITE, mutableListOf())
        detachAdv(attachedAdvPhones, now)
        detachSite(attachedSitePhones, now)
        lastTimeProperty.set(now)
    }

    private fun detachAdv(attachedAdvPhones: List<ClientPhone>, now: LocalDateTime) {
        logger.info("Find ${attachedAdvPhones.size} attached adv phones")
        attachedAdvPhones
            .asSequence()
            .mapNotNull { phone -> tryDetach(phone, now, TelephonyPhoneType.ADV) }
            .chunked(CHUNK_UPDATE_SIZE)
            .forEach {
                clientPhoneRepository.update(shard, it)
                telephonyPhoneService.resetStatusBsSynced(shard, getPhoneIdsIfTelephonyPhoneChanged(it))
            }
    }

    private fun detachSite(attachedSitePhones: MutableList<ClientPhone>, now: LocalDateTime) {
        logger.info("Find ${attachedSitePhones.size} attached on site phones")
        attachedSitePhones
            .asSequence()
            .mapNotNull { phone -> tryDetach(phone, now, TelephonyPhoneType.SITE) }
            .chunked(CHUNK_UPDATE_SIZE)
            .forEach { clientPhoneRepository.update(shard, it) }
    }

    /**
     * Попытаться оторвать номер Телефонии для [phone],
     * Для коллтрекинга в ТГО: если [last_show_time] телефона меньше чем [now] - [maxDaysWithoutShowsAdv].
     * Для коллтрекинга на сайте: если [last_show_time] телефона меньше чем [now] - [maxDaysWithoutActionsOnSite].
     * Для успешного отрыва вернутся изменения для заданного [phone], иначе `null`
     */
    private fun tryDetach(
        phone: ClientPhone,
        now: LocalDateTime,
        type: TelephonyPhoneType
    ): AppliedChanges<ClientPhone>? {
        val phoneId = phone.id
        val mc = ModelChanges(phoneId, ClientPhone::class.java)
        if (isTooEarlyToDetach(phone, now, type)) {
            logger.info("Too early to detach phone to $phoneId")
            return null
        }
        logger.info("Try to detach phone to $phoneId")
        try {
            telephonyPhoneService.detachTelephony(phone.telephonyServiceId, false)
            mc.process(null, ClientPhone.TELEPHONY_PHONE)
            mc.process(null, ClientPhone.TELEPHONY_SERVICE_ID)
            logger.info("Finish detach phone to $phoneId")
            return mc.applyTo(phone)
        } catch (e: RuntimeException) {
            logger.error("Failed to detach phone to $phoneId", e)
        }
        return null
    }

    private fun isTooEarlyToDetach(phone: ClientPhone, now: LocalDateTime, type: TelephonyPhoneType): Boolean {
        return type == TelephonyPhoneType.ADV
            && now.minusDays(maxDaysWithoutShowsAdv.toLong()).isBefore(phone.lastShowTime)
            || type == TelephonyPhoneType.SITE
            && now.minusDays(maxDaysWithoutActionsOnSite.toLong()).isBefore(phone.lastShowTime)
    }

    private fun isActualLastShowTimeCalculated(
        calcLastShowTimeJobLastTimeProperty: PpcProperty<LocalDateTime>?,
        now: LocalDateTime
    ): Boolean {
        val lastTime = calcLastShowTimeJobLastTimeProperty?.get() ?: return false
        return lastTime.plusDays(DEFAULT_DAYS_TELEPHONY_CALC_LAST_SHOW_TIME_JOB_NOT_STARTED).isAfter(now)
    }

    private fun isTimeToExecute(lastTimeProperty: PpcProperty<LocalDateTime>?, now: LocalDateTime?): Boolean {
        val lastTime = lastTimeProperty?.get() ?: return true
        return lastTime.plusDays(1).isBefore(now)
    }

}
