package ru.yandex.direct.jobs.telephony;

import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.TextFormat;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderCloseException;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.RetryingLogbrokerBatchReader;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.ess.common.logbroker.LogbrokerConsumerPropertiesImpl;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.ServiceNumberAction;

import static ru.yandex.direct.common.db.PpcPropertyNames.TELEPHONY_SERVICE_NUMBER_ACTION_PROCESSING_ENABLED;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_CALLTRACKING;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба для получения сообщений {@link ServiceNumberAction} о действиях с телефоном от Телефонии.
 * Сейчас поддержано только действие ReleaseAction -- отвязать старый номер и привязать новый.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 1),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, DIRECT_CALLTRACKING},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 15 * 60, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class TelephonyServiceNumberActionJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(TelephonyServiceNumberActionJob.class);

    static final String CONFIG_SECTION_NAME = "telephony_actions";
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final TelephonyPhoneService telephonyPhoneService;
    private final Runnable initializer;

    private boolean isInitialized = false;
    private LogbrokerBatchReader<ServiceNumberAction> logbrokerReader;

    @Autowired
    public TelephonyServiceNumberActionJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            DirectConfig directConfig,
            TvmIntegration tvmIntegration,
            TelephonyPhoneService telephonyPhoneService
    ) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.telephonyPhoneService = telephonyPhoneService;
        this.initializer = createInitializer(directConfig, tvmIntegration);
    }

    private Runnable createInitializer(DirectConfig directConfig, TvmIntegration tvmIntegration) {
        return () -> {
            var config = directConfig.getBranch(CONFIG_SECTION_NAME);
            var credentialsSupplier = createCredentialsSupplier(config, tvmIntegration);
            var logbrokerClientFactory = new LogbrokerClientFactoryFacade(credentialsSupplier);

            var consumerProperties = getConsumerProperties(config);
            var syncConsumerSupplier = logbrokerClientFactory.createConsumerSupplier(consumerProperties);

            logbrokerReader = new RetryingLogbrokerBatchReader<>(
                    () -> new TelephonyServiceNumberActionReader(syncConsumerSupplier, false),
                    consumerProperties.getRetries()
            );
            isInitialized = true;
        };
    }

    @Override
    public void execute() {
        if (!isJobEnabled()) {
            logger.info("Skip processing. Job is not enabled.");
            return;
        }
        initialize();
        try {
            logbrokerReader.fetchEvents(this::processEvents);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    @Override
    public void finish() {
        if (logbrokerReader != null) {
            try {
                this.logbrokerReader.close();
            } catch (LogbrokerReaderCloseException ex) {
                logger.error("Error while closing logbrokerReader", ex);
            }
        }
        isInitialized = false;
    }

    private void initialize() {
        if (!isInitialized) {
            logger.info("Initializing TelephonyServiceNumberActionJob");
            initializer.run();
        }
    }

    private void processEvents(List<ServiceNumberAction> events) {
        logger.info(
                "Fetched {} events: {}",
                events.size(),
                String.join(", ", mapList(events, TextFormat::shortDebugString))
        );
        if (events.isEmpty()) {
            return;
        }
        var serviceNumberIdsForReassign = StreamEx.of(events)
                .filter(ServiceNumberAction::hasRelease)
                .map(ServiceNumberAction::getServiceNumberID)
                .toSet();
        logger.info("Start reassigning telephony number for service number ids {}", serviceNumberIdsForReassign);
        var obsoleteServiceNumberIds =
                telephonyPhoneService.reassignTelephonyPhonesByServiceIds(serviceNumberIdsForReassign);
        logger.info("End reassigning telephony number for service number ids {}", obsoleteServiceNumberIds);
    }

    private boolean isJobEnabled() {
        return ppcPropertiesSupport.get(TELEPHONY_SERVICE_NUMBER_ACTION_PROCESSING_ENABLED).getOrDefault(false);
    }

    private static LogbrokerConsumerPropertiesImpl getConsumerProperties(DirectConfig config) {
        return new LogbrokerConsumerPropertiesImpl.Builder()
                .setHost(config.getString("host"))
                .setReadDataTimeoutSec(config.getLong("data_timeout"))
                .setInitTimeoutSec(config.getLong("init_timeout"))
                .setRetries(config.getInt("retries"))
                .setGroups(Collections.singletonList(config.getInt("group")))
                .setReadTopic(config.getString("topic"))
                .setConsumerName(config.getString("client_id"))
                .build();
    }

    private static Supplier<Credentials> createCredentialsSupplier(
            DirectConfig config,
            TvmIntegration tvmIntegration
    ) {
        var logbrokerTvmServiceName = config.getString("tvm_service_name");
        var logbrokerTvmService = TvmService.fromStringStrict(logbrokerTvmServiceName);

        return () -> {
            var serviceTicket = tvmIntegration.getTicket(logbrokerTvmService);
            return Credentials.tvm(serviceTicket);
        };
    }
}
