package ru.yandex.direct.jobs.telephony;

import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingPhone;
import ru.yandex.direct.core.entity.calltrackingphone.repository.CalltrackingPhoneRepository;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.telephony.client.TelephonyClient;
import ru.yandex.telephony.backend.lib.proto.telephony_platform.ServiceNumber;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_CALLTRACKING;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;

/**
 *  Синхронизирует с телефонией пул номеров используемых для коллтрекинга
 *  Удаляет из пула номера которые не обновлялись
 *  больше {@link PpcPropertyNames#CALLTRACKING_SYNC_PHONES_POOL_DELETE_OLDER_THAN_DAYS} дней
 *  Запускается раз в сутки
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 1),
        needCheck = ProductionOnly.class,
        //PRIORITY: Временно поставили приоритет по умолчанию;
        tags = {DIRECT_PRIORITY_1_NOT_READY, DIRECT_CALLTRACKING},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
// Время выбрано произвольно. Можно менять
@Hourglass(cronExpression = "10 10 2 * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class TelephonySyncCalltrackingPhonesPoolJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(TelephonySyncCalltrackingPhonesPoolJob.class);

    private static final int DEFAULT_BATCH_SIZE = 1000;
    private static final int DEFAULT_DELETE_OLDER_THAN_DAYS = 180;

    private final CalltrackingPhoneRepository calltrackingPhoneRepository;
    private final TelephonyClient telephonyClient;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    public TelephonySyncCalltrackingPhonesPoolJob(
            CalltrackingPhoneRepository calltrackingPhoneRepository, TelephonyClient telephonyClient,
            PpcPropertiesSupport ppcPropertiesSupport) {
        this.calltrackingPhoneRepository = calltrackingPhoneRepository;
        this.telephonyClient = telephonyClient;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    @Override
    public void execute() {
        LocalDateTime now = LocalDateTime.now();
        syncPool(now);
        deleteOldRecords(now);
    }

    private void syncPool(LocalDateTime now) {
        int offset = 0;
        int batchSize = getBatchSize();

        while (syncBatch(now, offset, batchSize) == batchSize) {
            offset += batchSize;
        }
    }

    /**
     *  Метод синхронизирует пачку номеров телефонии
     *  Возвращает количество синхронизированных методом номеров
     */
    private int syncBatch(LocalDateTime now, int offset, int count) {
        List<ServiceNumber> batch = telephonyClient.getServiceNumbersBatch(offset, count);
        List<CalltrackingPhone> models = batch.stream()
                .map(serviceNumber -> new CalltrackingPhone()
                        .withPhone(serviceNumber.getNum())
                        .withLastUpdate(now))
                .collect(Collectors.toList());

        calltrackingPhoneRepository.addOrUpdate(models);

        return models.size();
    }

    /**
     *  Метод удаляет записи старше
     *  чем {@link PpcPropertyNames#CALLTRACKING_SYNC_PHONES_POOL_DELETE_OLDER_THAN_DAYS} дней
     *  Возвращает количество удаленных методом номеров
     */
    private void deleteOldRecords(LocalDateTime now) {
        int deleteOlderThanDays = getDeleteOlderThanDays();
        LocalDateTime olderThanDateTime = now.minusDays(deleteOlderThanDays);
        int deletedCalltrackingPhones = calltrackingPhoneRepository.deleteOlderThan(olderThanDateTime);
        logger.info("Deleted {} old calltracking phones", deletedCalltrackingPhones);
    }

    private int getBatchSize() {
        return ppcPropertiesSupport.get(PpcPropertyNames.CALLTRACKING_SYNC_PHONES_POOL_BATCH_SIZE)
                .getOrDefault(DEFAULT_BATCH_SIZE);
    }

    private int getDeleteOlderThanDays() {
        return ppcPropertiesSupport.get(PpcPropertyNames.CALLTRACKING_SYNC_PHONES_POOL_DELETE_OLDER_THAN_DAYS)
                .getOrDefault(DEFAULT_DELETE_OLDER_THAN_DAYS);
    }
}
