package ru.yandex.direct.jobs.trackingphone;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.jobs.trackingphone.model.UnlinkData;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static ru.yandex.direct.common.db.PpcPropertyNames.CALLTRACKING_UNLINK_PHONE_ENABLED;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба отвязки подменных телефонов для баннеров с организациями, на которые у клиента нет прав.
 * Список баннеров определяется запросом {@link trackingphone/GetClientPhonesToUnlink.sql}
 * Если нужно отвязать номер Телефонии, то также освобождаем его и удаляем.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2), needCheck = ProductionOnly.class,
        //PRIORITY: Временно поставили приоритет по умолчанию; maxlog
        tags = {DIRECT_PRIORITY_1_NOT_READY})
@Hourglass(cronExpression = "0 0 1 * * ?", needSchedule = TypicalEnvironment.class)
public class UnlinkTrackingPhoneJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(UnlinkTrackingPhoneJob.class);

    private static final YtCluster CLUSTER = YtCluster.HAHN;
    private static final String CLIENT_ID_FIELD = "ClientID";
    private static final String CLIENT_PHONE_ID_FIELD = "client_phone_id";

    private static final String YQL_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("trackingphone/GetClientPhonesToUnlink.sql").readLines()
    );

    private final YtProvider ytProvider;
    private final ClientPhoneRepository clientPhoneRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final ShardHelper shardHelper;
    private final TelephonyPhoneService telephonyPhoneService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final FeatureService featureService;

    @Autowired
    public UnlinkTrackingPhoneJob(YtProvider ytProvider,
                                  ShardHelper shardHelper,
                                  ClientPhoneRepository clientPhoneRepository,
                                  BannerCommonRepository bannerCommonRepository,
                                  TelephonyPhoneService telephonyPhoneService,
                                  PpcPropertiesSupport ppcPropertiesSupport,
                                  FeatureService featureService) {
        this.ytProvider = ytProvider;
        this.shardHelper = shardHelper;
        this.clientPhoneRepository = clientPhoneRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.telephonyPhoneService = telephonyPhoneService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.featureService = featureService;
    }

    @Override
    public void execute() {

        boolean jobEnabled = ppcPropertiesSupport.get(CALLTRACKING_UNLINK_PHONE_ENABLED).getOrDefault(false);
        if (!jobEnabled) {
            logger.info("Skip processing, job is not enabled");
            return;
        }

        List<UnlinkData> unlinkDataAll = ytProvider.getOperator(CLUSTER, YtSQLSyntaxVersion.SQLv1)
                .yqlQuery(YQL_QUERY, rs -> new UnlinkData()
                        .withClientId(rs.getLong(CLIENT_ID_FIELD))
                        .withClientPhoneId(rs.getLong(CLIENT_PHONE_ID_FIELD))
                );

        Set<ClientId> ids = listToSet(unlinkDataAll, data -> ClientId.fromLong(data.getClientId()));
        Map<ClientId, Boolean> featureEnabled = featureService.isEnabledForClientIdsOnlyFromDb(
                ids, FeatureName.CALLTRACKING_UNLINK_PHONE_ALLOWED.getName());

        List<UnlinkData> unlinkData = filterList(unlinkDataAll, data -> {
            ClientId clientId = ClientId.fromLong(data.getClientId());
            return featureEnabled.get(clientId);
        });

        if (unlinkData.isEmpty()) {
            return;
        }

        logger.info("Data to unlink: {}", unlinkData);

        shardHelper.groupByShard(unlinkData, ShardKey.CLIENT_ID, UnlinkData::getClientId)
                .forEach((shard, data) -> {
                    List<Long> clientPhoneIds = mapList(unlinkData, UnlinkData::getClientPhoneId);

                    unlinkPhones(shard, clientPhoneIds);
                    freeTelephonyNumbers(shard, clientPhoneIds);

                    List<Long> clientIds = mapList(unlinkData, UnlinkData::getClientId);
                    clientPhoneRepository.delete(shard, clientIds, clientPhoneIds);
                });
    }

    private void unlinkPhones(int shard, List<Long> clientPhoneIds) {

        Map<Long, List<Long>> phonesToBanners =
                clientPhoneRepository.getBannerIdsByPhoneId(shard, clientPhoneIds);
        List<Long> bidsToUnlink = flatMap(phonesToBanners.values(), Function.identity());

        logger.info("Banner ids to unlink phones: {}", bidsToUnlink);

        clientPhoneRepository.unlinkBannerPhonesByBannerId(shard, bidsToUnlink);
        bannerCommonRepository.resetStatusBsSyncedByIds(shard, bidsToUnlink);

        Map<Long, List<Long>> phonesToCampaigns =
                clientPhoneRepository.getCampaignIdsByPhoneId(shard, clientPhoneIds);
        List<Long> cidsToUnlink = flatMap(phonesToCampaigns.values(), Function.identity());

        logger.info("Campaign ids to unlink phones: {}", cidsToUnlink);

        clientPhoneRepository.unlinkCampaignPhonesByCampaignId(shard, cidsToUnlink);
    }

    private void freeTelephonyNumbers(int shard, List<Long> clientPhoneIds) {

        List<ClientPhone> telephonyPhones = clientPhoneRepository.getTelephony(shard, clientPhoneIds);

        Set<String> serviceIdsToUnlink = listToSet(telephonyPhones, ClientPhone::getTelephonyServiceId);
        serviceIdsToUnlink.forEach(telephonyPhoneService::detachTelephony);
    }
}
