package ru.yandex.direct.jobs.trustedredirects.job;

import java.time.LocalDate;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.db.PpcPropertyType;
import ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.trustedredirects.service.TrustedRedirectsYtUploader;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;

/**
 * Выгружает все записи из таблицы ppcdict.trusted_redirects в реплицируюмую таблицу в YT markov.[//home/direct/export/trusted_domain]
 * <p>
 * Выгрузка выполняется раз в день
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1_NOT_READY, CheckTag.GROUP_INTERNAL_SYSTEMS}
)
@Hourglass(cronExpression = "0 15 * * * ?", needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class ExportTrustedRedirectsToYtJob extends DirectJob {
    static final PpcPropertyName<LocalDate> LAST_EXECUTION =
            new PpcPropertyName<>("export_trusted_redirects_last_execution", PpcPropertyType.LOCAL_DATE);
    static final YtCluster EXPORT_CLUSTER = YtCluster.MARKOV;

    private static final Logger logger = LoggerFactory.getLogger(ExportTrustedRedirectsToYtJob.class);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final YtProvider ytProvider;
    private final TrustedRedirectsService trustedRedirectsService;

    @Autowired
    public ExportTrustedRedirectsToYtJob(PpcPropertiesSupport ppcPropertiesSupport, YtProvider ytProvider,
                                         TrustedRedirectsService trustedRedirectsService) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.ytProvider = ytProvider;
        this.trustedRedirectsService = trustedRedirectsService;
    }

    @Override
    public void execute() {
        if (!shouldRun()) {
            logger.info("Skipping current run, job had already been performed today");
            return;
        }

        TrustedRedirectsYtUploader uploader = new TrustedRedirectsYtUploader(ytProvider,
                EXPORT_CLUSTER, trustedRedirectsService.getCounterTrustedRedirects()
        );
        uploader.upload();

        LocalDate now = LocalDate.now();
        PpcProperty<LocalDate> property = ppcPropertiesSupport.get(LAST_EXECUTION);
        logger.info("Saving {} to {} property", now, property.get());
        property.set(now);
    }

    /**
     * Вернуть истину, если джоба может быть запущена
     */
    boolean shouldRun() {
        LocalDate lastExecutionTime = ppcPropertiesSupport.get(LAST_EXECUTION).get();
        logger.info("{} property value is {}", LAST_EXECUTION, lastExecutionTime);
        return lastExecutionTime == null
                || !LocalDate.now().equals(lastExecutionTime);
    }
}
