package ru.yandex.direct.jobs.trustedredirects.service;

import java.util.Arrays;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.trustedredirects.model.RedirectType;
import ru.yandex.direct.core.entity.trustedredirects.model.TrustedRedirects;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.proxy.ApiServiceClient;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Класс выполняющий непосредственную загрузку выбранных данных в динамическую таблицу в YT
 */
@ParametersAreNonnullByDefault
public class TrustedRedirectsYtUploader {
    private static final String TRUSTED_DOMAIN_EXPORT_PATH = "export/trusted_domain";
    private static final String DOMAIN = "domain";
    private static final String REDIRECT_TYPE = "redirect_type";
    private static final String OPTS = "opts";

    private static final TableSchema schema = new TableSchema.Builder()
            .addKey(DOMAIN, ColumnValueType.STRING)
            .addKey(REDIRECT_TYPE, ColumnValueType.STRING)
            .addKey(OPTS, ColumnValueType.STRING)
            .build();

    private static final TableSchema keySchema = new TableSchema.Builder()
            .addKey(DOMAIN, ColumnValueType.STRING)
            .build();

    private static final Logger logger = LoggerFactory.getLogger(TrustedRedirectsYtUploader.class);

    private final YtProvider ytProvider;
    private final YtDynamicOperator operator;
    private final String path;
    private final List<TrustedRedirects> trustedRedirects;

    public TrustedRedirectsYtUploader(YtProvider ytProvider, YtCluster exportCluster,
                                      List<TrustedRedirects> trustedRedirects) {
        YtClusterConfig ytClusterConfig = ytProvider.getClusterConfig(exportCluster);

        this.ytProvider = ytProvider;
        this.operator = ytProvider.getDynamicOperator(exportCluster);
        this.path = YtPathUtil.generatePath(ytClusterConfig.getHome(), relativePart(), TRUSTED_DOMAIN_EXPORT_PATH);
        this.trustedRedirects = trustedRedirects;
    }

    public void upload() {
        logger.info("Selecting replicas");
        YTreeNode replica = operator.runRpcCommand(client -> {
            return Iterables.getLast(client.getNode(path + "/@replicas").join() // IGNORE-BAD-JOIN DIRECT-149116
                    .asMap().values(), null);
        });
        YtCluster cluster =
                YtCluster.valueOf(replica.mapNode().getString("cluster_name").replace("-", "_").toUpperCase());
        String readPath = replica.mapNode().getString("replica_path");

        logger.info("Will read data from {} in cluster {}", readPath, cluster);
        YtDynamicOperator readOperator = ytProvider.getDynamicOperator(cluster);
        List<UnversionedRow> tableContents = readOperator.runRpcCommand(client -> {
            return readTableContents(client, readPath);
        });

        logger.info("Updating trusted redirects table in main cluster");
        operator.runInTransaction(transaction -> replaceWithNewData(transaction, tableContents));
        logger.info("Successfully finished updating");
    }

    private List<UnversionedRow> readTableContents(ApiServiceClient client, String readPath) {
        UnversionedRowset tableContents = client.selectRows(
                String.format("%s FROM [%s]", DOMAIN, readPath)).join(); // IGNORE-BAD-JOIN DIRECT-149116
        logger.info("Got {} rows from table {}", tableContents.getRows().size(), readPath);
        return tableContents.getRows();
    }

    private boolean replaceWithNewData(ApiServiceTransaction transaction, List<UnversionedRow> currentRows) {
        // Можно удалять только те данные, которые отсутствуют в новом чанке, но в нашем случае данных очень мало
        // и проще все удалить и залить заново
        ModifyRowsRequest delReq = new ModifyRowsRequest(path, keySchema);
        for (UnversionedRow row : currentRows) {
            if (row != null) {
                delReq.addDelete(mapList(row.getValues(), UnversionedValue::getValue));
            }
        }
        delReq.setRequireSyncReplica(false);

        ModifyRowsRequest insertReq = new ModifyRowsRequest(path, schema);
        for (TrustedRedirects rec : trustedRedirects) {
            String setString = RepositoryUtils.setToDb(rec.getOpts(), Enum::toString);
            insertReq.addInsert(
                    Arrays.asList(rec.getDomain(), RedirectType.toSource(rec.getRedirectType()).getLiteral(),
                            setString == null ? "" : setString));
        }

        insertReq.setRequireSyncReplica(false);

        logger.info("Deleting all rows from table");
        transaction.modifyRows(delReq).join(); // IGNORE-BAD-JOIN DIRECT-149116
        logger.info("Inserting new rows in database");
        transaction.modifyRows(insertReq).join(); // IGNORE-BAD-JOIN DIRECT-149116
        return true;
    }
}
