package ru.yandex.direct.jobs.turbolandings;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes;
import ru.yandex.direct.dbqueue.LimitOffset;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Удаление устаревших задач (архивных и нет) по установке прав доступа к счетчикам метрики.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 4),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_2, JOBS_RELEASE_REGRESSION},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_MAXLOG,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 3600, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdateCounterGrantsDbQueueCleaner extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(UpdateCounterGrantsDbQueueCleaner.class);
    private static final LimitOffset LIMIT = LimitOffset.maxLimited();
    private static final Duration LIFETIME = Duration.ofDays(3);

    private final DbQueueRepository dbQueueRepository;

    @Autowired
    public UpdateCounterGrantsDbQueueCleaner(DbQueueRepository dbQueueRepository) {
        this.dbQueueRepository = dbQueueRepository;
    }

    @Override
    public void execute() {
        logger.debug("Going to delete jobs older than {}", LIFETIME);
        int r = dbQueueRepository.cleanup(getShard(), DbQueueJobTypes.UPDATE_COUNTER_GRANTS_JOB, LIFETIME, LIMIT);
        logger.info("Deleted {} jobs", r);
    }
}
