package ru.yandex.direct.jobs.turbolandings;

import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.turbolanding.container.UpdateCounterGrantsParams;
import ru.yandex.direct.core.entity.turbolanding.container.UpdateCounterGrantsParamsItem;
import ru.yandex.direct.core.entity.turbolanding.container.UpdateCounterGrantsResult;
import ru.yandex.direct.core.entity.turbolanding.service.UpdateCounterGrantsService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbqueue.JobFailedWithTryLaterException;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.service.DbQueueService;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.utils.FunctionalUtils;

import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.UPDATE_COUNTER_GRANTS_JOB;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_0, JOBS_RELEASE_REGRESSION}
)
@Hourglass(periodInSeconds = 60, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdateMetrikaCounterGrantsJob extends DirectShardedJob {
    private static final Duration GRAB_FOR = Duration.ofMinutes(15);
    private static final int MAX_JOBS_PER_RUN = 50;
    private static final double DEVIATION = 0.3;

    // При ошибке делаем перезапуск задания, до 3х раз с нарастающим интервалом: 5, 15, 40 минут
    private static final Map<Long, Duration> RUN_AFTER_BY_TRYCOUNT =
            ImmutableMap.<Long, Duration>builder()
                    .put(1L, Duration.ofMinutes(5))
                    .put(2L, Duration.ofMinutes(15))
                    .put(3L, Duration.ofMinutes(40))
                    .build();
    private static final int MAX_ATTEMPTS = RUN_AFTER_BY_TRYCOUNT.size() + 1;

    private static final Logger logger = LoggerFactory.getLogger(UpdateMetrikaCounterGrantsJob.class);

    private final DbQueueService dbQueueService;
    private final UpdateCounterGrantsService updateCounterGrantsService;
    private final UserService userService;

    @Autowired
    public UpdateMetrikaCounterGrantsJob(DbQueueService dbQueueService,
                                         UpdateCounterGrantsService updateCounterGrantsService,
                                         UserService userService) {
        this.dbQueueService = dbQueueService;
        this.updateCounterGrantsService = updateCounterGrantsService;
        this.userService = userService;
    }

    /**
     * Конструктор нужен только для тестов. Используется для указания шарда.
     */
    UpdateMetrikaCounterGrantsJob(int shard,
                                  DbQueueService dbQueueService,
                                  UpdateCounterGrantsService updateCounterGrantsService,
                                  UserService userService) {
        super(shard);
        this.dbQueueService = dbQueueService;
        this.updateCounterGrantsService = updateCounterGrantsService;
        this.userService = userService;
    }

    @Override
    public void execute() {
        int shard = getShard();
        for (int i = 0; i < MAX_JOBS_PER_RUN; i++) {

            boolean foundJob = dbQueueService.grabAndProcessJob(shard, UPDATE_COUNTER_GRANTS_JOB, GRAB_FOR,
                    this::processGrabbedJob, MAX_ATTEMPTS, this::handleProcessingException);

            if (!foundJob) {
                break;
            }
        }
    }

    private UpdateCounterGrantsResult processGrabbedJob(
            DbQueueJob<UpdateCounterGrantsParams, UpdateCounterGrantsResult> jobInfo) {
        UpdateCounterGrantsParams jobArgs = jobInfo.getArgs();
        logger.info("Start processing jobId={}, attempt={}", jobInfo.getId(), jobInfo.getTryCount());

        List<UpdateCounterGrantsParamsItem> items = jobArgs.getItems();

        Map<Long, String> uidToLogin = getUidToLoginMap(items);

        Map<Long, Set<String>> userIdsByCounterId = StreamEx.of(items)
                .toMap(UpdateCounterGrantsParamsItem::getCounterId,
                        item -> convertUidsToLogins(item.getUserIds(), uidToLogin));

        UpdateCounterGrantsService.UpdateCounterGrantsResultForRetry result =
                updateCounterGrantsService.updateCounterGrants(userIdsByCounterId);

        if (result.isNeedToRetry()) {
            UpdateCounterGrantsParams jobArgsToRetry =
                    getArgsToRetry(jobArgs, result.getCounterIdsThatShouldNotRetry());
            Duration runAfter = getRandomRunAfter(jobInfo.getTryCount());

            logger.warn("error in updateCounterGrants for userIdsByCounterId={}, job={}", userIdsByCounterId, jobInfo);
            throw new JobFailedWithTryLaterException(runAfter, jobArgsToRetry);
        }

        logger.info("Finish processing jobId={}, attempt={}", jobInfo.getId(), jobInfo.getTryCount());
        return UpdateCounterGrantsResult.success();
    }

    private Map<Long, String> getUidToLoginMap(List<UpdateCounterGrantsParamsItem> items) {
        Set<Long> userIds = StreamEx.of(items)
                .map(UpdateCounterGrantsParamsItem::getUserIds)
                .flatMap(Collection::stream)
                .toSet();
        Collection<User> users = userService.massGetUser(userIds);
        return StreamEx.of(users).toMap(User::getUid, User::getLogin);
    }

    private Set<String> convertUidsToLogins(Collection<Long> uids, Map<Long, String> uidToLogin) {
        return StreamEx.of(uids)
                .map(uidToLogin::get)
                .nonNull().toSet();
    }

    private UpdateCounterGrantsResult handleProcessingException(
            DbQueueJob<UpdateCounterGrantsParams, UpdateCounterGrantsResult> jobInfo, String stacktrace) {
        return UpdateCounterGrantsResult.error(stacktrace);
    }

    private Duration getRandomRunAfter(Long jobTryCount) {
        Duration runAfter = RUN_AFTER_BY_TRYCOUNT.get(jobTryCount);
        if (runAfter == null) {
            return null;
        }
        long runAfterInMillis = runAfter.toMillis();

        long randomRunAfterInMillis = ThreadLocalRandom.current().nextLong(
                (long) (runAfterInMillis * (1.0 - DEVIATION)),
                (long) (runAfterInMillis * (1.0 + DEVIATION)));

        return Duration.ofMillis(randomRunAfterInMillis);
    }

    private static UpdateCounterGrantsParams getArgsToRetry(UpdateCounterGrantsParams originalJobArgs,
                                                            Set<Long> counterIdsThatShouldNotRetry) {
        return new UpdateCounterGrantsParams()
                .withItems(FunctionalUtils.filterList(originalJobArgs.getItems(),
                        item -> !counterIdsThatShouldNotRetry.contains(item.getCounterId())));
    }
}
