package ru.yandex.direct.jobs.uac

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes
import ru.yandex.direct.core.entity.uac.model.CheckAudienceSegmentJobParams
import ru.yandex.direct.core.entity.uac.model.CheckAudienceSegmentJobResult
import ru.yandex.direct.dbqueue.model.DbQueueJob
import ru.yandex.direct.dbqueue.repository.DbQueueRepository
import ru.yandex.direct.dbqueue.service.DbQueueService
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.jobs.uac.service.UacCheckAudienceSegmentsJobService
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob
import java.time.Duration
import java.time.LocalDateTime

@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 10), needCheck = NonDevelopmentEnvironment::class,
    tags = [CheckTag.DIRECT_PRIORITY_2],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_BABASHEV],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 180, needSchedule = NonDevelopmentEnvironment::class)
class CheckAudienceSegmentsJob @Autowired constructor(
    private val dbQueueService: DbQueueService,
    private val dbQueueRepository: DbQueueRepository,
    private val uacCheckAudienceSegmentsJobService: UacCheckAudienceSegmentsJobService,
) : DirectShardedJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(CheckAudienceSegmentsJob::class.java)
        private const val JOB_RETRIES = 1
        private val ITERATION_TIME = Duration.ofMinutes(5)
        private val PERIOD_BETWEEN_TRIES = Duration.ofMinutes(5)
    }

    override fun execute() {
        val borderTime = LocalDateTime.now().plus(ITERATION_TIME)

        do {
            val grabbed = dbQueueService.grabAndProcessJob(
                shard,
                DbQueueJobTypes.UAC_CHECK_AUDITION_SEGMENTS,
                { processGrabbedJobWrapped(it) },
                JOB_RETRIES,
                { _, stacktrace -> onError(stacktrace) }
            )
            if (!grabbed) logger.info("No free jobs")
        } while (grabbed && LocalDateTime.now() < borderTime)
    }

    private fun processGrabbedJobWrapped(
        jobInfo: DbQueueJob<CheckAudienceSegmentJobParams, CheckAudienceSegmentJobResult>
    ): CheckAudienceSegmentJobResult {
        try {
            return processGrabbedJob(jobInfo)
        } catch (e: Exception) {
            if (e is InterruptedException) {
                Thread.currentThread().interrupt()
            }
            throw e
        }
    }

    private fun processGrabbedJob(
        jobInfo: DbQueueJob<CheckAudienceSegmentJobParams, CheckAudienceSegmentJobResult>
    ): CheckAudienceSegmentJobResult {
        val checkedCampaign = uacCheckAudienceSegmentsJobService
            .checkCampaignsAudienceSegments(jobInfo.args.uacCampaignId, jobInfo.clientId, jobInfo.uid)

        if (checkedCampaign?.audienceSegmentsSynchronized != true) {
            dbQueueRepository.insertDelayedJob(
                shard,
                DbQueueJobTypes.UAC_CHECK_AUDITION_SEGMENTS,
                jobInfo.clientId,
                jobInfo.uid,
                jobInfo.args,
                PERIOD_BETWEEN_TRIES
            )
            return CheckAudienceSegmentJobResult("Campaign is not ready")
        }

        return CheckAudienceSegmentJobResult(null)

    }

    private fun onError(stacktrace: String): CheckAudienceSegmentJobResult {
        return CheckAudienceSegmentJobResult(stacktrace)
    }
}
